% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/density_area.R
\name{density_polygons}
\alias{density_polygons}
\title{Density polygons}
\usage{
density_polygons(
  x,
  y,
  probs = 0.5,
  as_sf = FALSE,
  as_list = FALSE,
  range_mult = 0.25,
  rangex = NULL,
  rangey = NULL,
  ...
)
}
\arguments{
\item{x, y}{Numeric data dimensions}

\item{probs}{Probabilities to compute density polygons for}

\item{as_sf}{Should the returned values be \link[sf:sf]{sf::sf}? Defaults to \code{FALSE}.}

\item{as_list}{Should the returned value be a list? Defaults to \code{FALSE} to
work with \code{\link[dplyr:reframe]{dplyr::reframe()}}}

\item{range_mult}{A multiplier to the range of \code{x} and \code{y} across which the
probability density will be estimated.}

\item{rangex, rangey}{Custom ranges across \code{x} and \code{y} ranges across which the
probability density will be estimated.}

\item{...}{Additional arguments to be passed to \code{\link[ggdensity:get_hdr]{ggdensity::get_hdr()}}}
}
\value{
A list of data frames, if \code{as_list=TRUE}, or just a data frame,
if \code{as_list=FALSE}.
\subsection{Data frame output}{

If \code{as_sf=FALSE}, the data frame has the following columns:
\describe{
\item{level_id}{An integer id for each probability level}
\item{id}{An integer id for each sub-polygon within a probabilty level}
\item{prob}{The probability level (originally passed to \code{probs})}
\item{x, y}{The values along the original \code{x} and \code{y} dimensions defining
the density polygon. These will be renamed to the original input variable
names.}
\item{order}{The original plotting order of the polygon points, for
convenience.}
}
}

\subsection{sf output}{

If \code{as_sf=TRUE}, the data frame has the following columns:
\describe{
\item{level_id}{An integer id for each probability level}
\item{prob}{The probability level (originally passed to \code{probs})}
\item{geometry}{A column of \code{\link[sf:st]{sf::st_polygon()}}s.}
}

This output will need to be passed to \code{\link[sf:sf]{sf::st_sf()}} to utilize many of the
features of \link[sf:sf]{sf::sf}.
}
}
\description{
Given numeric vectors \code{x} and \code{y}, \code{density_polygons()} will return
a data frame, or list of a data frames, of the polygon defining 2d kernel
densities.
}
\details{
When using \code{density_polygons()} together with \code{\link[dplyr:summarise]{dplyr::summarise()}}, \code{as_list}
should be \code{TRUE}.

If both \code{rangex} and \code{rangey} are defined, \code{range_mult} will be disregarded.
If only one or the other of \code{rangex} and \code{rangey} are defined, \code{range_mult}
will be used to produce the range of the undefined one.
}
\examples{
library(densityarea)
library(dplyr)
library(purrr)
library(sf)

ggplot2_inst <- require(ggplot2)
tidyr_inst <- require(tidyr)

set.seed(10)
x <- c(rnorm(100))
y <- c(rnorm(100))

# ordinary data frame output
poly_df <- density_polygons(x,
                            y,
                            probs = ppoints(5))

head(poly_df)

# It's necessary to specify a grouping factor that combines `level_id` and `id`
# for cases of multimodal density distributions
if(ggplot2_inst){
  ggplot(poly_df, aes(x, y)) +
    geom_path(aes(group = paste0(level_id, id),
                  color = prob))
}

# sf output
poly_sf <- density_polygons(x,
                            y,
                            probs = ppoints(5),
                            as_sf = TRUE)

head(poly_sf)

# `geom_sf()` is from the `{sf}` package.
if(ggplot2_inst){
  poly_sf |>
    arrange(desc(prob)) |>
    ggplot() +
    geom_sf(aes(fill = prob))
}

# Tidyverse usage

data(s01)

# Data transformation
s01 <- s01 |>
  mutate(log_F1 = -log(F1),
         log_F2 = -log(F2))

## Basic usage with `dplyr::reframe()`
### Data frame output
s01 |>
  group_by(name) |>
  reframe(density_polygons(log_F2,
                           log_F1,
                           probs = ppoints(5))) ->
  speaker_poly_df

if(ggplot2_inst){
  speaker_poly_df |>
    ggplot(aes(log_F2, log_F1)) +
    geom_path(aes(group = paste0(level_id, id),
                  color = prob)) +
    coord_fixed()
}

### sf output
s01 |>
  group_by(name) |>
  reframe(density_polygons(log_F2,
                           log_F1,
                           probs = ppoints(5),
                           as_sf = TRUE)) |>
  st_sf() ->
  speaker_poly_sf

if(ggplot2_inst){
  speaker_poly_sf |>
    ggplot() +
    geom_sf(aes(color = prob),
            fill = NA)
}

## basic usage with dplyr::summarise()
### data frame output

if(tidyr_inst){
  s01 |>
    group_by(name) |>
    summarise(poly = density_polygons(log_F2,
                                      log_F1,
                                      probs = ppoints(5),
                                      as_list = TRUE)) |>
    unnest(poly) ->
    speaker_poly_df
}
### sf output

if(tidyr_inst){
  s01 |>
    group_by(name) |>
    summarise(poly = density_polygons(
      log_F2,
      log_F1,
      probs = ppoints(5),
      as_list = TRUE,
      as_sf = TRUE
    )) |>
    unnest(poly) |>
    st_sf() ->
    speaker_poly_sf
}
}
