

# guess_fileSize ---------------------------------------------------------------
#' @title Estimate the File Size of the Software
#'
#' @description The function makes a rough estimation of the size of the
#' R package using the base R function [file.size]. Files in `.Rbuildignore` are
#' exclude. Please note that this estimation does not necessarily reflect the installed size
#' of the package.
#'
#' @param root the root file path
#'
#' @param .ignore optional vector of regular expresssions that will be ignore when the file
#' size is guessed
#'
#' @seealso [base::file.size]
#'
#' @examples
#'
#' guess_fileSize()
#'
#' @md
#' @noRd
guess_fileSize <- function(root = ".", .ignore = NULL) {
  ## no root, no file size
  if (is.null(root))
    return(NULL)

  ## check for .Rbuildignore, everything listed should be excluded since
  ## it will not become part of the final package
  rbuildignore_path <- file.path(root, ".Rbuildignore")
  if (file.exists(rbuildignore_path) && is.null(.ignore)) {
    .ignore <- readLines(normalizePath(rbuildignore_path), warn = FALSE)
    .ignore <- .ignore[.ignore != ""] # eliminate blank lines

  } else {
    .ignore <- " "

  }

  ## grep all files of interest (exclude hidden files)
  ## winslash = "/" is used in all normalizePath() calls to allow gsub() matching.
  files <- normalizePath(
    list.files(
      path = normalizePath(root),
      recursive = TRUE,
      full.names = TRUE,
      all.files = FALSE
    ),
    winslash = "/"
  )

  ## make paths relative to root
  files <- gsub(paste0(normalizePath(root, winslash = "/"), "/"), "", files)

  ## exclude full dirs matching an ignore pattern
  dirs <- list.dirs(normalizePath(root, winslash = "/"))
  dirs <- gsub(paste0(normalizePath(root, winslash = "/"), "/"), "", dirs)
  ignore_dirs <- dirs[grepl(paste(.ignore, collapse = "|"), dirs, perl = TRUE)]
  files <- files[
    !grepl(paste0("^", ignore_dirs, collapse = "|"), files, perl = TRUE)
  ]

  ## kick-out all files that do not belong to the R package
  files <-
    files[!grepl(paste(.ignore, collapse = "|"), files, perl = TRUE)]

  ## estimate total size
  paste0(sum(file.size(files)) / 1e3, "KB")
}

