% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/format_weather.R
\name{format_weather}
\alias{format_weather}
\title{Format weather data into a standardised format.}
\usage{
format_weather(
  w,
  YYYY = NULL,
  MM = NULL,
  DD = NULL,
  hh = NULL,
  mm = NULL,
  ss = NULL,
  POSIXct_time = NULL,
  time_zone = NULL,
  temp,
  rain,
  rh,
  ws,
  wd,
  wd_sd,
  station,
  lon = NULL,
  lat = NULL,
  lonlat_file = NULL,
  data_check = TRUE
)
}
\arguments{
\item{w}{a \code{\link{data.frame}} object of weather station data for
formatting.}

\item{YYYY}{Column name \code{character} or index in \code{w} that refers to the year when the
weather was logged.}

\item{MM}{Column name \code{character} or index in \code{w} that refers to the month (numerical)
when the weather was logged.}

\item{DD}{Column name \code{character} or index in \code{w} that refers to the day of month when
the weather was logged.}

\item{hh}{Column name \code{character} or index in \code{w} that refers to the hour (24 hour) when
the weather was logged.}

\item{mm}{Column name \code{character} or index in \code{w} that refers to the minute when the
weather was logged.}

\item{ss}{Column name \code{character} or index in \code{w} that refers to the second when the
weather was logged.}

\item{POSIXct_time}{Column name \code{character} or index in \code{w} which contains a \code{POSIXct}
formatted time. This can be used instead of arguments \code{YYYY}, \code{MM}, \code{DD},
\code{hh}, \code{mm.}.}

\item{time_zone}{Time zone (Olsen time zone format) \code{character} where the
weather station is located. May be in a column or supplied as a character string.
Optional, see also \code{r}. See details.}

\item{temp}{Column name \code{character} or index in \code{x} that refers to temperature in degrees
Celsius.}

\item{rain}{Column name \code{character} or index in \code{w} that refers to rainfall in millimetres.}

\item{rh}{Column name \code{character} or index in \code{w} that refers to relative
humidity as a percentage.}

\item{ws}{Column name \code{character} or index in \code{w} that refers to wind speed in km / h.}

\item{wd}{Column name \code{character} or index in \code{w} that refers to wind direction in
degrees.}

\item{wd_sd}{Column name \code{character} or index in \code{w} that refers to wind speed columns
standard deviation.  This is only applicable if weather data
is already summarised to hourly increments. See details.}

\item{station}{Column name \code{character} or index in \code{w} that refers to the weather station
name or identifier. See details.}

\item{lon}{Column name \code{character} or index in \code{w} that refers to weather station's
longitude. See details.}

\item{lat}{Column name \code{character} or index in \code{w} that refers to weather station's
latitude. See details.}

\item{lonlat_file}{A file path (\code{character}) to a \acronym{CSV} which included station
name/id and longitude and latitude coordinates if they are not supplied in
the data. Optional, see also \code{lon} and \code{lat}.}

\item{data_check}{If \code{TRUE}, it checks for NA values in all 'rain', 'temp',
'rh', 'wd' and 'ws' data and if any values which are unlikely. Use a character
vector of variable names, (wither any of 'rain', 'temp', 'rh', 'wd' or 'ws')
to check data from specific variables. If FALSE it ignores all variables and
could cause subsequent models using this data to fail.}
}
\value{
A \code{epiphy.weather} object (an extension of \CRANpkg{data.table})
containing the supplied weather aggregated to each hour in a suitable
format for use with disease models. Depending on the input weather, classes
will be given to the output object to indicate which models it meets the data
requirements for. Some of the columns returned are as follows:
\tabular{rl}{
\strong{times}: \tab Time in POSIXct format with "UTC" time-zone\cr
\strong{rain}: \tab Rainfall in mm \cr
\strong{temp}: \tab Temperature in degrees Celsius \cr
\strong{ws}: \tab Wind speed in km / h \cr
\strong{wd}: \tab Wind direction in compass degrees \cr
\strong{wd_sd}: \tab Wind direction standard deviation in compass degrees \cr
\strong{lon}: \tab Station longitude in decimal degrees \cr
\strong{lat}: \tab Station latitude in decimal degrees \cr
\strong{station}: \tab Unique station identifying name \cr
\strong{YYYY}: \tab Year \cr
\strong{MM}: \tab Month \cr
\strong{DD}: \tab Day \cr
\strong{hh}: \tab Hour \cr
\strong{mm}: \tab Minute \cr}
}
\description{
Formats raw weather data into an object suitable for use in the affiliated
spore dispersal packages such as \code{ascotraceR}. This
standardised data format ensures that the supplied weather data meet the
requirements for functions in the aforementioned packages. Input weather
data expects a 'long' data format, where each line is an observation at
a point in time.
}
\details{
\code{time_zone} The time-zone in which the \code{time} was recorded. All weather
stations in \code{w} must fall within the same time-zone.  If the required stations
are located in differing time zones, \code{format_weather()} should be run separately
on each object, then data can be combined after formatting.

\code{wd_sd} If weather data is
provided in hourly increments, a column
with the standard deviation of the wind direction over the hour is required
to be provided. If the weather data are sub-hourly, the standard deviation
will be calculated and returned automatically.

\code{lon}, \code{lat} and \code{lonlat_file} If \code{w} provides longitude and
latitude values for station locations, these may be specified in the \code{lon}
and \code{lat} columns.  If the coordinates are not relevant to the study
location \code{NA} can be specified and the function will drop these column
variables.  If these data are not included, (\code{NULL}) a separate file may be
provided that contains the longitude, latitude and matching station name to
provide station locations in the final \code{epiphy.weather} object that is
created by specifying the file path to a \acronym{CSV} file using
\code{lonlat_file}.
}
\examples{
# load the weather data to be formatted
weather_station_data <-
   read.csv(system.file("extdata",
                        "scaddan_weather.csv",
                        package = "cercospoRa"))
weather_station_data$Local.Time <-
   as.POSIXct(weather_station_data$Local.Time, format = "\%Y-\%m-\%d \%H:\%M:\%S",
              tz = "UTC")

weather <- format_weather(
   w = weather_station_data,
   POSIXct_time = "Local.Time",
   ws = "meanWindSpeeds",
   wd_sd = "stdDevWindDirections",
   rain = "Rainfall",
   temp = "Temperature",
   wd = "meanWindDirections",
   lon = "Station.Longitude",
   lat = "Station.Latitude",
   station = "StationID",
   time_zone = "UTC"
)

# Reformat saved weather

# Create file path and save data
file_path_name <- paste(tempdir(), "weather_saved.csv", sep = "\\\\")
write.csv(weather, file = file_path_name,
          row.names = FALSE)

# Read data back in to
weather2 <- read.csv(file_path_name, stringsAsFactors = FALSE)

# reformat the data to have appropriate column classes and data class
weather2 <- format_weather(weather2,
                           time_zone = "UTC")
unlink(file_path_name) # remove temporary weather file
}
