% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/network.comm.num.R
\name{network.comm.num}
\alias{network.comm.num}
\title{Consistent Estimation of the Number of Communities via Regularized Network Embedding.}
\usage{
network.comm.num(A, sample.index.n, lambda, Z.int, B.int,
                        a=3, kappa=1, alpha=1, eps=5e-2, niter=20,
               niter.Z=5, update.B="ADMM",local.oppro=FALSE, merge.all=TRUE,
               ad.BIC=FALSE, Fully.Connected=TRUE, trace=FALSE,
               line.search=TRUE, ad.BIC.B=FALSE)
}
\arguments{
\item{A}{An observed n * n adjacency matrix of undirected graph.}

\item{sample.index.n}{A 3 * (n*(n-1)/2) matrix, all pairs of integers from 1 to n.}

\item{lambda}{A list, the sequences of the tuning parameters (\eqn{\lambda_1}{\lambda_1}, \eqn{\lambda_2}{\lambda_2}, and \eqn{\lambda_3}{\lambda_3}).}

\item{Z.int}{A n * r matrix, the initial values of embedding vectors corresponding to n nodes.}

\item{B.int}{A n * r matrix, the initial values of community centers corresponding to n nodes.}

\item{a}{A float value, regularization parameter in MCP, the default setting is 3.}

\item{kappa}{A float value, the penalty parameter in ADMM algorithm, the default setting is 1.}

\item{alpha}{A float value, the step size of coordinate descent algorithm updating Z, the default setting is 1.}

\item{eps}{A float value, algorithm termination threshold.}

\item{niter}{Int, maximum number of cycles of the overall ADMM algorithm.}

\item{niter.Z}{Int, maximum number of cycles of coordinate descent algorithm updating Z.}

\item{update.B}{The optimization algorithm updating B, which can be selected "ADMM" (default) and "AMA".}

\item{local.oppro}{The logical variable, whether to use local approximations when updating Z, the default setting is F.}

\item{merge.all}{Whether to merge pairs of nodes indirectly connected (but without the direct edge) in the estimated community membership matrix.}

\item{ad.BIC}{Whether to use the adjusted BIC, the default setting is F.}

\item{Fully.Connected}{Whether to use the all pairs (i,j) in fusion penalty, the default setting is T. If F, the pairs (i,j) in fusion penalty will be determined by the observed n * n adjacency matrix A.}

\item{trace}{Whether to output the intermediate process of the algorithm.}

\item{line.search}{Linear search or not, the default setting is T.}

\item{ad.BIC.B}{Whether the BIC criterion contains terms involving the B matrix, the default setting is F.}
}
\value{
A list including all estimated parameters and the BIC values with all choices of given tuning parameters, and the selected optional parameters.
Opt_Z: A n * r matrix, the estimated embedding vectors corresponding to n nodes;
Opt_B: A n * r matrix, the estimated community centers corresponding to n nodes;
Opt_K: Int, the estimated number of communities;
Opt_member: A n-dimensional vector, describing the membership of n nodes;
Opt_cluster.matrix: A n * n membership matrix, whose (i,j)-element is 1, if nodes i and j belong to the same community, and 0, otherwise.
}
\description{
The main function for Consistent Estimation of the Number of Communities via Regularized Network Embedding.
}
\examples{
\donttest{
library(cencrne)
data(example.data)
A                   = example.data$A
K.true              = example.data$K.true
Z.true              = example.data$Z.true
B.true              = example.data$B.true
P.true              = example.data$P.true
Theta.true          = example.data$Theta.true
cluster.matrix.true = example.data$cluster.matrix.true

n       = dim(A)[1]
lam.max = 3
lam.min = 0.5
lam1.s  = 2/log(n)
lam2.s  = sqrt(8*log(n)/n)
lam3.s  = 1/8/log(n)/sqrt(n)
lambda  = genelambda.obo(nlambda1=3,lambda1_max=lam.max*lam1.s,lambda1_min=lam.min*lam1.s,
                         nlambda2=10,lambda2_max=lam.max*lam2.s,lambda2_min=lam.min*lam2.s,
                         nlambda3=1,lambda3_max=lam.max*lam3.s,lambda3_min=lam.min*lam3.s)

sample.index.n = rbind(combn(n,2),1:(n*(n-1)/2))
int.list       = gen.int(A)
Z.int          = int.list$Z.int
B.int          = int.list$B.int
res            = network.comm.num(A, sample.index.n, lambda, Z.int, B.int)

K.hat = res$Opt_K # the estimated number of communities
Z.hat = res$Opt_Z # the estimated embedding vectors corresponding to n nodes
cluster.matrix.hat = res$Opt_cluster.matrix # the n * n estimated membership matrix
evaluation(Z.hat, Z.true, cluster.matrix.hat, cluster.matrix.true,
           P.true, Theta.true, K.hat, K.true)
}

}
\references{
Ren, M., Zhang S. and Wang J. (2022). Consistent Estimation of the Number of Communities via Regularized Network Embedding.
}
\author{
Mingyang Ren, Sanguo Zhang, Junhui Wang. Maintainer: Mingyang Ren \href{mailto:renmingyang17@mails.ucas.ac.cn}{renmingyang17@mails.ucas.ac.cn}.
}
