#functions: g_raw, g_ecs, g_tcs, g_mcr, g_difp.
#methods: plot.callback_stat

#'
#' Raw callback rates plot
#'
#' @param data A \code{callback_stat} object.
#' @param col  A list of two colors.
#' @param ... further arguments passed to or from other methods.
#'
#' @return A ggplot2 object
#'
#' @author Emmanuel Duguet
#'
#' @examples
#' data(origin1)
#' m <- callback(labour1,"offer","hist","callback","all")
#' s <- stat_raw(m)
#' g_raw(s)
#'
#' @importFrom ggplot2 ggplot aes geom_bar coord_flip scale_x_discrete
#' geom_errorbar ggtitle ylab xlab theme element_text
#'
#' @export

g_raw <-function(data = NULL, col = NULL, ...){
  if (is.null(col)==TRUE) {col <- c("#619CFF","#606060")}
  method <- as.character(data$specif["method"])
  m_name <- as.character(data$specif["confid"])
  level  <- as.numeric(data$specif["level"])
  cand <- p_callback <- inf <- sup <- NULL
  pr <- data[["props"]]
  gr <- data.frame(
    cand = rownames(pr),
    p_callback = pr$p_callback,
    inf = pr$inf_p_callback,
    sup = pr$sup_p_callback
  )
  
  ggplot(data = gr, aes(x = cand, y = p_callback)) +
    geom_bar(stat = "identity", fill = col[1]) +
    coord_flip() +
    scale_x_discrete(limits = rev) +
    geom_errorbar(aes(ymin = inf, ymax = sup),
                  width = 0.2,
                  color = col[2]) +
    ggtitle("raw callback rates") +
    ylab(paste(
      m_name,
      "confidence intervals at the",
      round(level * 100, 1),
      "percent level"
    )) +
    xlab("Candidates") +
    theme(
      plot.title = element_text(
        color = "gray50",
        size = 14,
        face = "bold"
      ),
      axis.title.y = element_text(
        color = "black",
        size = 12,
        face = "plain"
      ),
      axis.title.x = element_text(
        color = "black",
        size = 10,
        face = "plain"
      )
    )
}

#'
#' Exclusive callback shares plot
#'
#' @param data a \code{callback_stat} object.
#' @param col  A list of four colors.
#' @param ... further arguments passed to or from other methods.
#'
#' @return A ggplot2 object
#'
#' @author Emmanuel Duguet
#'
#' @examples
#' data(labour1)
#' m <- callback(labour1,"offer","hist","callback","all")
#' s <- stat_ecs(m)
#' g_ecs(s)
#'
#' @importFrom ggplot2 ggplot aes geom_bar coord_flip geom_errorbar ggtitle
#' ylab xlab theme element_text scale_color_manual scale_x_continuous sec_axis
#' guides scale_fill_manual geom_hline
#'
#' @importFrom stats relevel
#'
#' @export

g_ecs <- function(data = NULL, col = NULL, ...) {
  if (is.null(col)==TRUE) {col <- c("#F8766D","#619CFF","#808080","#000000")}
  candidate <-
    r <- inf1 <- sup1 <- c10 <- disc <- c01 <- p_cand1 <- NULL
  inf_p_cand1 <-
    inf_p_cand2 <- sup_p_cand1 <- sup_p_cand2 <- dif_p_value <-
    NULL
  method <- as.character(data$specif["method"])
  m_name <- as.character(data$specif["confid"])
  level  <- as.numeric(data$specif["level"])
  pr  <- data[["props"]]
  gr <- data.frame(
    cand = rownames(pr),
    r1 = pr$p_cand1,
    inf1 = pr$inf_p_cand1,
    sup1 = pr$sup_p_cand1,
    r2 = pr$p_cand2
  )
  x <- strsplit(gr$cand, ".vs.")
  l_sub <- function(x, n) {
    x[n]
  }
  gr$cand1 <- sapply(x, l_sub, 1)
  gr$cand2 <- sapply(x, l_sub, 2)
  
  gr1 <- gr[, c("cand1", "r1", "inf1", "sup1")]
  colnames(gr1) <- c("cand", "r", "inf1", "sup1")
  gr1$t <- 1:nrow(gr1)
  gr1$candidate <- "cand1"
  
  gr2 <- gr[, c("cand2", "r2")]
  colnames(gr2) <- c("cand", "r")
  gr2$t <- 1:nrow(gr2)
  gr2$candidate <- "cand2"
  
  ngr <- rbind(gr1[, c("t", "candidate", "cand", "r")], gr2)
  ngr$candidate <- factor(ngr$candidate)
  ngr$candidate <- relevel(ngr$candidate, ref = "cand2")
  
  ggplot(data = ngr, aes(x = t, group = candidate, fill = candidate)) +
    coord_flip() +
    scale_color_manual(values = col[1:2]) +
    scale_fill_manual(values = col[1:2]) +
    geom_bar(stat = "identity", aes(y = r)) +
    scale_x_continuous(
      breaks = 1:length(gr1$cand),
      labels = gr1$cand,
      trans = "reverse",
      sec.axis = sec_axis(
        ~ .,
        breaks = 1:length(gr2$cand),
        labels = gr2$cand
      )
    ) +
    geom_errorbar(
      data = gr1,
      aes(x = t, ymin = inf1, ymax = sup1),
      width = 0.3,
      color = col[3]
    ) +
    geom_hline(yintercept = 0.5, color = col[4]) +
    ggtitle("share of exclusive callbacks") +
    ylab(paste(
      m_name,
      "confidence intervals at the",
      round(level * 100, 1),
      "percent level"
    )) +
    xlab("") +
    theme(
      plot.title = element_text(
        color = "gray50",
        size = 14,
        face = "bold"
      ),
      axis.title.y = element_text(
        color = "black",
        size = 12,
        face = "plain"
      ),
      axis.title.x = element_text(
        color = "black",
        size = 10,
        face = "plain"
      ),
      axis.text.y = element_text(colour = col[2]),
      axis.text.y.right = element_text(color = col[1])
    ) +
    guides(fill = "none", color = "none")
}

#'
#' Total callback shares plot
#'
#' @param data a \code{callback_stat} object.
#' @param col  A list of three colors.
#' @param ... further arguments passed to or from other methods.
#'
#' @return A ggplot2 object
#'
#' @author Emmanuel Duguet
#'
#' @examples
#' data(labour1)
#' m <- callback(labour1,"offer","hist","callback","all")
#' s <- stat_tcs(m)
#' g_tcs(s)
#'
#' @importFrom ggplot2 ggplot aes geom_bar coord_flip geom_errorbar ggtitle
#' ylab xlab theme element_text scale_color_manual scale_x_continuous sec_axis
#' guides scale_fill_manual geom_hline
#'
#' @importFrom stats relevel
#'
#' @export

g_tcs <- function(data = NULL, col = NULL, ...) {
  if (is.null(col)==TRUE) {col <- c("#C0C0C0","#619CFF","#F8766D")}
  callback <-
    r <- inf1 <- sup1 <- c10 <- disc <- c01 <- p_cand1 <- NULL
  inf_p_cand1 <-
    inf_p_cand2 <- sup_p_cand1 <- sup_p_cand2 <- dif_p_value <-
    NULL
  
  pr <- data[["props"]]
  gr <- data.frame(
    cand = rownames(pr),
    r1 = pr$p_cand1,
    r2 = pr$p_cand2,
    r3 = pr$p_equal
  )
  
  x <- strsplit(gr$cand, ".vs.")
  l_sub <- function(x, n) {
    x[n]
  }
  gr$cand1 <- sapply(x, l_sub, 1)
  gr$cand2 <- sapply(x, l_sub, 2)
  gr$cand3 <- "both"
  
  gr1 <- gr[, c("cand1", "r1")]
  colnames(gr1) <- c("cand", "r")
  gr1$t <- 1:nrow(gr1)
  gr1$callback <- "1 only"
  
  gr2 <- gr[, c("cand2", "r2")]
  colnames(gr2) <- c("cand", "r")
  gr2$t <- 1:nrow(gr2)
  gr2$callback <- "2 only"
  
  gr3 <- gr[, c("cand3", "r3")]
  colnames(gr3) <- c("cand", "r")
  gr3$t <- 1:nrow(gr3)
  gr3$callback <- "both"
  
  ngr <- rbind(gr1, gr2[, c("t", "callback", "cand", "r")], gr3)
  ngr$callback <- factor(ngr$callback)
  ngr$callback <-
    ordered(ngr$callback, levels = c("both", "2 only", "1 only"))
  
  ggplot(data = ngr, aes(x = t, group = callback, fill = callback)) +
    coord_flip() +
    scale_color_manual(values = col[1:3]) +
    scale_fill_manual(values = col[1:3]) +
    geom_bar(stat = "identity", aes(y = r)) +
    scale_x_continuous(
      breaks = 1:length(gr1$cand),
      labels = gr1$cand,
      trans = "reverse",
      sec.axis = sec_axis(
        ~ .,
        breaks = 1:length(gr2$cand),
        labels = gr2$cand
      )
    ) +
    ggtitle("share of total callbacks") +
    xlab("") +
    theme(
      plot.title = element_text(
        color = "gray50",
        size = 14,
        face = "bold"
      ),
      axis.title.y = element_text(
        color = "black",
        size = 12,
        face = "plain"
      ),
      axis.title.x = element_text(
        color = "black",
        size = 10,
        face = "plain"
      ),
      axis.text.y = element_text(colour = col[3]),
      axis.text.y.right = element_text(color = col[2])
    )
}

#'
#' Proportions' comparison plot
#'
#' @param data A \code{callback_stat} object.
#' @param col  A list of three colors.
#' @param ... further arguments passed to or from other methods.
#'
#' @return A ggplot2 object
#'
#' @author Emmanuel Duguet
#'
#' @examples
#' data(labour1)
#' m <- callback(labour1,"offer","hist","callback","all")
#' s <- stat_mcr(m)
#' g_mcr(data = s)
#'
#' @importFrom ggplot2 ggplot aes geom_bar coord_flip geom_errorbar ggtitle
#' ylab xlab theme element_text scale_x_continuous sec_axis guides
#'
#' @export

g_mcr <- function(data = NULL, col = NULL, ...) {
  if (is.null(col)==TRUE) {col <- c("#619CFF","#F8766D","#606060")}
  p_cand <-
    candidate <- inf <- sup <- inf_p_callback <- inf_p_cand1 <- NULL
  inf_p_cand2 <-
    sup_p_callback <- sup_p_cand1 <- sup_p_cand2 <- dif_p_value <- NULL
  method <- as.character(data$specif["method"])
  m_name <- as.character(data$specif["confid"])
  level  <- as.numeric(data$specif["level"])
  pr <- data[["props"]]
  gr <- data.frame(
    cand = rownames(pr),
    p_cand1 = pr$p_cand1,
    inf1 = pr$inf_p_cand1,
    sup1 = pr$sup_p_cand1,
    p_cand2 = pr$p_cand2,
    inf2 = pr$inf_p_cand2,
    sup2 = pr$sup_p_cand2
  )
  
  x <- strsplit(gr$cand, ".vs.")
  l_sub <- function(x, n) {
    x[n]
  }
  gr$cand1 <- sapply(x, l_sub, 1)
  gr$cand2 <- sapply(x, l_sub, 2)
  
  gr1 <- gr[, c("cand1", "inf1", "p_cand1", "sup1")]
  colnames(gr1) <- c("cand", "inf", "p_cand", "sup")
  
  gr1$t <- 1:nrow(gr1)
  gr1$candidate <- "cand1"
  gr2 <- gr[, c("cand2", "inf2", "p_cand2", "sup2")]
  colnames(gr2) <- c("cand", "inf", "p_cand", "sup")
  
  gr2$t <- 1:nrow(gr2)
  gr2$candidate <- "cand2"
  ngr <- rbind(gr1, gr2)
  
  ggplot(data = ngr, aes(
    x = t,
    y = p_cand,
    group = candidate,
    colour = candidate
  )) +
    coord_flip() +
    scale_color_manual(values = col[1:2]) +
    scale_x_continuous(
      breaks = 1:length(gr1$cand),
      labels = gr1$cand,
      trans = "reverse",
      sec.axis = sec_axis(
        ~ .,
        breaks = 1:length(gr2$cand),
        labels = gr2$cand
      )
    ) +
    geom_errorbar(aes(ymin = inf, ymax = sup), width = 0.3) +
    ggtitle("Matched callback rates") +
    ylab(paste(
      m_name,
      "confidence intervals at the",
      round(level * 100, 1),
      "percent level"
    )) +
    xlab("") +
    theme(
      plot.title = element_text(
        color = "gray50",
        size = 14,
        face = "bold"
      ),
      axis.title.y = element_text(
        color = "black",
        size = 12,
        face = "plain"
      ),
      axis.title.x = element_text(
        color = "black",
        size = 10,
        face = "plain"
      ),
      axis.text.y = element_text(colour = col[1]),
      axis.text.y.right = element_text(color = col[2])
    )  +
    guides(color = "none")
}

#'
#' Difference of proportions plot
#'
#' @param data A \code{callback_stat} object.
#' @param col  A list of three colors.
#' @param ... further arguments passed to or from other methods.
#'
#' @return A ggplot2 object
#'
#' @author Emmanuel Duguet
#'
#' @examples
#' data(labour1)
#' m <- callback(labour1,"offer","hist","callback","all")
#' s <- stat_mcr(m)
#' g_difp(s)
#'
#' @importFrom ggplot2 ggplot aes geom_bar coord_flip geom_errorbar ggtitle
#' ylab xlab theme element_text scale_x_continuous sec_axis guides geom_hline
#'
#' @export

g_difp <- function(data = NULL, col = NULL, ...) {
  if (is.null(col)==TRUE) {col <- c("#619CFF","#F8766D","#808080")}
  method <- data$specif["method"]
  if (method=="cp") {cat("Plot not available with \"cp\", use \"student\" or \"wilson\" instead ")
  } else {
  cand <- candidate <- inf <- sup <- p_cand <- NULL
  m_name <- as.character(data$specif["confid"])
  method <- as.character(data$specif["method"])
  level  <- as.numeric(data$specif["level"])
  pr <- data[["props"]]
  gr <- data.frame(
    cand = rownames(pr),
    inf = pr$inf_cand_dif,
    sup = pr$sup_cand_dif,
    p_cand = pr$p_cand_dif
  )
  
  grt <- data$specif["convention"]
  
  x <- strsplit(gr$cand, ".vs.")
  l_sub <- function(x, n) {
    x[n]
  }
  gr$cand1 <- sapply(x, l_sub, 1)
  gr$cand2 <- sapply(x, l_sub, 2)
  gr$t <- 1:nrow(gr)
  
  ggplot(data = gr, aes(x = t, y = p_cand)) +
    coord_flip() +
    scale_color_manual(values = col[1:2]) +
    scale_x_continuous(
      breaks = 1:length(gr$cand1),
      labels = gr$cand1,
      trans = "reverse",
      sec.axis = sec_axis(
        ~ .,
        breaks = 1:length(gr$cand2),
        labels = gr$cand2
      )
    ) +
    geom_errorbar(aes(ymin = inf, ymax = sup),
                  width = 0.3,
                  colour = col[3]) +
    geom_hline(yintercept = 0,
               linetype = "dashed",
               colour = "gray50") +
    ggtitle(grt) +
    ylab(paste(
      m_name,
      "confidence intervals at the",
      round(level * 100, 1),
      "percent level"
    )) +
    xlab("") +
    theme(
      plot.title = element_text(
        color = "gray50",
        size = 14,
        face = "bold"
      ),
      axis.title.y = element_text(
        color = "black",
        size = 12,
        face = "plain"
      ),
      axis.title.x = element_text(
        color = "black",
        size = 10,
        face = "plain"
      ),
      axis.text.y = element_text(colour = col[1]),
      axis.text.y.right = element_text(color = col[2])
    ) +
    guides(color = "none")
  }#else end
}

#'
#' Plots for callback rates and shares
#'
#' @param x a \code{callback_stat} object.
#' @param col  A list of four colors.
#' @param dif  FALSE for the confidence intervals (the default), TRUE for the 
#' difference in proportions
#' @param ... further arguments passed to or from other methods.
#'
#' @return a ggplot2 object
#'
#' @author Emmanuel Duguet
#'
#' @export
#'
#' @examples
#' data(labour1)
#' m <- callback(labour1,"offer","hist","callback","all")
#' s <- stat_mcr(m)
#' plot(s)
#'
plot.callback_stat <- function(x,
                          col = NULL,
                          dif = TRUE,
                          ...) {
  cid <- as.character(x$specif["cid"])
  if (cid=="raw") {
     g_raw(data = x, col = col)
     } else {
  if (dif == TRUE) {
    g_difp(data = x, col = col)
    } else {
    if (cid=="mcr") {
      g_mcr(data = x, col = col)
    } else if (cid=="tcs") {
      g_tcs(data = x, col = col)
    } else if (cid=="ecs") {g_ecs(data = x, col = col)}
  }
  }
}

