
# ------------------------------------------------------------------------
# 
# "thetaARRB<-function(X)" 
# 
# Compute the Extremal Index for non-Atomic Harris Markov Chains Using Pseudo-Blocks Division
#
# ------------------------------------------------------------------------
##'

#' @aliases thetaARRB
#' @title Compute the Extremal Index for Non-Atomic Markov Chains Using Pseudo-Regenerative
#' Blocks
#' @description This function divides the input dataset into pseudo-blocks for 
#' a non-atomic Markov chain 
#' using a Nummelin splitting trick with estimated parameters.
#' We use the optimal small set computed by \code{\link{findBestEpsilon}} 
#' function and calculates the extremal index using quantile-based thresholds.
#' @param X A numeric vector representing the Markov chain.
#' 
#' @details The function uses \code{\link{GetPseudoBlocks}} to divide \code{X} into blocks 
#' using to the estimated Nummelin splitting trick. 
#' High quantiles from \code{X} are generated and used as thresholds to compute 
#' statistics on each block, including sub-maximums 
#' and block indices. These statistics are then used to calculate the extremal index,
#' a measure of extreme value clustering, across the blocks.
#' @return A numeric vector representing the extremal index at various quantile thresholds.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#' Bertail, P. and Clémençon, S. (2006).
#' Regeneration-based statistics for Harris recurrent
#'   Markov chains.  
#' \emph{Lecture notes in Statistics}, vol. \bold{187}, pp. 1-54, Springer.
#' 
#' @seealso \code{\link{GetPseudoBlocks}}, \code{\link{fastNadaraya}},
#' \code{\link{regenboot}}, \code{\link{smallEnsemble}},\code{\link{findBestEpsilon}}.
#' @keywords "Approximate Regenerative Block" "Markov chains" "Small set" "Extremal index"
#' @export
#' @examples 
#' \donttest{
#' coeff=0.75
#' X = arima.sim(n=200, list(ar = c(coeff)))
#' thetaB <- thetaARBB(X)
#' plot(thetaB)
#' }

thetaARBB<-function(X)
{
  # find best small-set
  # Compute corresponding estimator of the kernel density estimator of the transition of time series 
  med = median(X)
  sm <- findBestEpsilon(X,s=med,plotIt=FALSE)
  h = bandw1(X)
  f=sm$trans   
  eps = sm$epsilon
  delta = sm$delta
  m = sm$s
  
  # Compute pseudo-regeneration blocks and the value of the maximum on each blocks 
  
  RES=GetPseudoBlocks(X, m, eps_opt = eps, delta_opt = delta, p_XiXip1 = f,func=max) # no truncation
  
  # retrieve daset, block decomposition and number of blocks
  dataset=RES[[1]]
  Block=RES[[2]]
  l_n=RES[[3]]
  
  # Take high level quantiles ( the extremal index is computed as a function of u)
  perc=seq(from=0.9,by=0.001,length.out=100)
  u=quantile(X,perc)
  # compute statistics on blocks
  Bnumber=Block[,1]
  lB=Block[,2]
  Submax=Block[,3]
  n=length(X)
  n_u=length(u)
  n_b=length(Submax)
  # compute extremal index based on blocks
  
  thetaB=(colSums((matrix(rep(Submax,n_u),n_b,n_u)>matrix(rep(u,n_b),n_b,n_u,byrow=TRUE)))
          /apply (rbind(matrix(1,1,n_u), colSums((matrix(rep(X,n_u),n,n_u)>matrix(rep(u,n),n,n_u,byrow=TRUE)))),2,max))
  return(thetaB)
}

