## ------------------------------------------------------------------------
# 
# "tft_boot(X, n, func, h, kernel, B, option)" 
# 
# Time-Frequency Transformation (TFT) Bootstrap
#
# ------------------------------------------------------------------------
##


# TFT bootstrap as decribed in Kirch and Politis (), see chapter 10, section 8
# Three possible options "local" for local bootstrap resampling, "wild" for wild bootstrap and 
#"res" for residuals of standardized periodogram
# Parameters 
# X time series
# n by default is the length of time series but allow for a smaller sample size m to perform moon bootstrap
# h a bandwith , by default of order n^{-2/3}
# kernel , only two possible choices kernel=0: daniell; else: bartlett-priestley (epanechnikov)
# B number of bootstrap replications 
# option can be "local", "wild" and "res"

#' @aliases tft_boot
#' @title TFT Bootstrap.
#' @description This function implements the Time Frequency Toggle (TFT)-Bootstrap
#' introduced in Kirch and Politis (2011).
#' @param X A numeric vector representing a time series.
#' @param n An integer; by default is the length of time series \code{X} but allow for a 
#' smaller sample size m to perform moon bootstrap.
#' @param func A function to be applied to each column of the TFT bootstrap samples.
#' @param h A numeric value specifying the bandwidth used to compute the kernel estimator 
#' in case of local bootstrap. 
#' By default it is equal to \eqn{n^{-2/3}}.
#' @param kernel An integer value indicating the kernel type. Use \code{0} 
#' for the Daniell kernel or any other value for the Bartlett-Priestley (Epanechnikov) kernel 
#' (by default).
#' @param B An integer indicating the number of bootstrap replications.
#' @param option A character string specifying the bootstrap method:
#' \code{"local"} for local bootstrap, \code{"wild"} for wild bootstrap, 
#' and \code{"res"} for residual bootstrap.
#' 
#' @details The function performs the Time Frequency Toggle (TFT)-Bootstrap. 
#' Its basic idea is to bootstrap the Fourier coefficients of the observed
#' time series, and then back-transforming them to obtain a bootstrap sample in the
#' time domain (see Kirch and Politis (2011)).
#' Then it computes 
#' the specified function, `func`, on each bootstrap replicate.
#'  Depending on the `option`, different bootstrap 
#' methods are used to construct bootstrap samples:
#' * `"local"` - local bootstrap.
#' * `"wild"`-  wild bootstrap.
#' * `"res"` - residual bootstrap.
#' 
#' @return An object of class `"boodd"`.
#' @author We are grateful to Claudia Kirch for providing the original code in \code{R}.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#' Kirch, C. and Politis, D. N. (2011). TFT-Bootstrap: Resampling time series in the frequency
#'  domain to obtain replicates in the time domain. \emph{Annals of Statistics}, vol. 
#'  \bold{39}, pp. 1427-1470
#' @seealso \code{\link{boot_local}}, \code{\link{boot_res}},
#' \code{\link{boot_wild}}.
#' @keywords Bootstrap "Frequency and time domain" "Fast Fourier Transform"
#' @export
#' @examples 
#' n=2000
#' phi0=0.5
#' p0=length(phi0)
#' theta0=0
#' q0=length(theta0)
#' sim <- arima.sim(n , model=list(ar = phi0, ma = theta0))
#' # The function to be bootstrapped using tft_boot (both variance and acf of order 1)
#' vc<-function(X){return(c(var(X),acf(X,plot=FALSE)$acf[2]))}
#' coeff=vc(sim)
#' bb1=tft_boot(sim,n=n,func=vc,B=999,option="local")
#' bb2=tft_boot(sim,n=n,func=vc,B=999,option="res")
#' bb3=tft_boot(sim,n=n,func=vc,B=999,option="wild")


tft_boot<-function(X,n=length(X),func,h=length(X)^{-2/3},kernel=1,B,option="local")
{
  # according to option create tft bootstrap samples
  if (option %in% c("local","wild","res")) {
    if (option=="local") {bb=boot_local(X,n=n,h=h,kernel,B)}
    if (option=="wild") {bb=boot_wild(X,n=n,h=h,kernel,B)}
    if (option=="res") {bb=boot_res(X,n=n,h=h,kernel,B)}
  }
  
  else message(" Only local res or wild are valid for option")
  
  # Apply function func to each columns of the tft bootstrap samples
  Wboot=t(apply(bb,MARGIN=2,FUN=func))
  # compute statistics on original data 
  coeff=func(X)
  # Return an object of class "boodd"
  resboot <- list(s = Wboot, Tn =coeff)
  class(resboot) <- "boodd"
  attr(resboot, "kind") <- "TFT"
  attr(resboot, "func") <- func
  return(resboot)
}


