##
# ------------------------------------------------------------------------
#
# "rate.block.sub(X, func, B=999, n_b=99, q=0.75, PLT=TRUE, ...)" --
#
# Block subsampling for time series with rate of convergence estimation
#
# ------------------------------------------------------------------------
##
#' @aliases rate.block.sub
#' @title Block Subsampling for Time Series with Convergence Rate Estimation
#' @description Performs block subsampling for a time series with an estimation of the convergence rate
#' of a given statistic. 
#' The function constructs subsampling distributions of specified sizes, estimates the convergence rate 
#' and asymptotic bias, and provides optional diagnostic plots.
#' @param X A numeric vector or a time series.
#' @param func The function to be estimated using block subsampling.
#' @param n_b A positive integer; the number of subsampling distributions to be constructed, 
#' with block sizes ranging from \eqn{\lfloor n^{1/3}\rfloor} to \eqn{\rfloor n^{3/4}\rfloor}, where \eqn{n} 
#' is the sample size.
#' @param q A numeric value in the interval \eqn{(0.5, 1)}; it is used to calculate the interquantile 
#' range which estimates the rate of convergence.
#' @param PLT A logical value; By default it is \code{TRUE}, plots graphics and intermediate 
#' regression results.
#' @param ... Additional arguments for the \code{func} function.
#' @details The function \code{rate.block.sub} performs block subsampling, specifically 
#' suited for time series data, estimates the convergence rate \eqn{n^{-\alpha}} and the approximation
#' rate \eqn{n^{-\beta}} between the true distribution and the asymptotic one.
#' 
#'
#' @return A list containing:
#' \item{alpha}{The estimated rate of convergence.}
#' \item{beta}{The estimated approximation rate.}
#' \item{obj}{An object of class \code{boodd} with the optimized subsampling distribution.}
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted..
#'             
#' Bertail, P.,  Politis, D., Romano, J. (1999). 
#' On Subsampling Estimators with Unknown Rate of Convergence.
#' \emph{ Journal of the American Statistical Association}, \bold{94}, 569-579.
#' 
#' @seealso \code{\link{rate.sub}}, \code{\link{best.block.sub.size}}, 
#' \code{\link{best.sub.size.iid}}, \code{\link{block.sub}}, \code{\link{boots}}.
#' @keywords Subsampling "Convergence rate estimation" 
#' @export
#' @examples 
#' set.seed(12345)
#' n = 500 # sample size
#' ts = arima.sim(n=n,model=list(ar=c(0.4)))*sqrt(1-0.4^2) # Generating an AR(1) 
#' # process with variance 1
#' # Moon MBB without replacement with estimated rate
#' reb=rate.block.sub(ts,mean,PLT=TRUE)


rate.block.sub<-function(X, func, n_b=99, q=0.75, PLT=TRUE, ...) {
  
  N=length(X)
  TN=func(X,...)
  s = N^(1/(n_b-1)*4/7)
  l_b = floor(N^(1/3)*c(1,cumprod(rep(s,n_b-1))))
  n_b = length(l_b)
  
  if ((q<=1/2) | (q>=1)) stop("Higher quantile for interquantile range estimation should be bigger than 1/2")
  
  q2=q
  q1=1-q
  T_matrix=matrix(0,n_b,N)
  # Estimation of the rate of convergence n^alpha,  that is  alpha
  
  qq=matrix(0,n_b,3) # computation of quantiles of subsampling distribution resp. median , q1 and q2 quantiles
  
  for (i in 1:n_b) {
    T_matrix[i,]=block.sub(X,func,l_b[i],method="circular",...)
    qq[i,]=quantile(T_matrix[i,],c(0.5,q1,q2))
  }
  # Plot data and print results of estimation only of PLT=TRUE
  llb=-log(l_b)
  diff=log(qq[,3]-qq[,2])
  rate_est=lm(diff~llb)
  if (PLT == TRUE) {
    oldpar <- par(no.readonly = TRUE)  
    on.exit(par(oldpar))               
    
    par(mfrow = c(1, 2))               
    plot(llb, diff)
    abline(rate_est$coeff, col = "red")
    summary(rate_est)
  }
  
  alpha=lm(log(qq[,3]-qq[,2])~llb)$coeff[2]
  
  # Estimation of bias : rate of convergence to asymptotic ( beta) 
  
  datareg= matrix(0,nrow=n_b/2, ncol=2)
  l1=floor(0.7*N)
  l2=floor(0.8*N)+1
  for (i in 1:(n_b/2)) {
    i1=i
    i2 =i+n_b/2
    
    # Recenter and standardize the subsampling distribution (with the range)
    # Use range instead of variance to have more robust standardisation
    
    Z1 = (T_matrix[i1,]-qq[i1,1])/sd(T_matrix[i1,])
    Z2 = (T_matrix[i2,]-qq[i2,1])/sd(T_matrix[i2,])
    
    a=t(cbind(rbind(Z1,1),rbind(Z2,2)))
    a=a[order(a[,1]),]
    Kb_n1n2 = (cumsum(a[,2]==2)-cumsum(a[,2]==1))/(2*N)
    
    # Mean at several point close to 0.75 quantile
    
    MM=mean(log(abs(Kb_n1n2[l1:l2])))
    datareg[i,]=c(MM,llb[i])
    
  }
  
  d=data.frame(yy=datareg[,1],xx=datareg[,2])
  reg = lm(yy ~xx, data=d,subset = !apply(d, 1, is.bad))
  beta = coef(reg)[2]
  
  if (PLT == TRUE) { 
    oldpar <- par(no.readonly = TRUE)  
    on.exit(par(oldpar)) 
    
    plot(d$xx,d$yy)
    T1 = seq(min(d$xx),max(d$xx),length=100)
    grille = data.frame(xx = T1)
    # intervalles de confiance sur les 100 valeurs 
    ICdte = predict(reg,newdata=grille,interval="confidence",level=0.95)
    # confidence intervals for slope
    matlines(grille$xx,cbind(ICdte),lty=c(1,2,2),col=1) 
    par(mfrow=c(1,1))
  }
  # determination of the closest subsampling distribution, close to asymptotically optimal choice of b
  b_opt=N^(1/(2*abs(alpha)+1))
  ind=which.min(abs(l_b-b_opt))
  # reconstruction of the subsampling distribution on the right scale (so that confint.boodd is valid
  sub_opt=sort(TN+(T_matrix[ind,]-qq[ind,1])*l_b[ind]^alpha/N^alpha)
  obj <- list(s = sub_opt, Tn = TN )
  class(obj) <- "boodd"
  attr(obj, "kind") <- "subsampling with rates estimation"
  attr(obj, "func") <- func
  
  names(alpha)="alpha"
  names(beta)="beta"                   
  return(list(alpha,beta,obj))
}


