## 
# ------------------------------------------------------------------------
# 
# "jackVarRegen(x,func,...,atom,small,s)" --
# 
# Jackknife estimator by regenerative blocks for the variance of function 'func'.
# 
# ------------------------------------------------------------------------
#' @aliases jackVarRegen
#' @title Jackknife Variance Estimator for Regenerative Processes
#' @description Estimates the variance of a statistic applied to a vector or 
#'              a matrix using a jackknife procedure tailored for regenerative processes,
#'              in particular recurrent Markov chains.
#'              
#' @param x A vector or a matrix representing the data.
#' @param func The function applied to each sample.
#' @param ... Optional additional arguments for the \code{func} function.
#' @param atom A numeric value or a string; an atom of the Markov chain in the atomic case.
#' @param small An optional object of class \code{\link{smallEnsemble}}. It can be
#' created optimally using \code{\link{findBestEpsilon}}.
#' @param s A real number specifying the center of the small set. Default is the median of \code{x}.
#' @details The \code{jackVarRegen} function is a versatile tool for estimating the jackknife variance 
#'          in cases of statistics based on regenerative blocks.
#'           It accommodates variable length blocks and is 
#'          effective for both finite state and general Markov chains.
#'          It calls \code{jackVarRegen.atom} and \code{jackVarRegen.smallEnsemble}, respectively in
#'          the atomic and the general case.
#' @return Returns a scalar or a covariance matrix, depending on whether the function \code{func} 
#' is univariate or multivariate. For a function returning a vector of length 
#' \emph{p}, the output will be a covariance matrix of size \emph{p x p}.
#' @references 
#' Bertail, P. and  Clémençon. S. (2006).
#' \emph{Regeneration-based statistics for Harris recurrent
#'   Markov chains}, 1-54.
#'  Lecture notes in Statistics 187. Springer.
#' 
#' Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#'
#'             
#' Quenouille, M.H. (1949). Approximate tests of correlation in time-series. 
#' \emph{J. Roy. Statist. Soc., Ser. B}, \bold{11}, 68-84.
#' 
#' Quenouille, M. H. (1956). Notes on bias in estimation , \emph{Biometrika}, \bold{43},
#'   353–360.
#' 
#' 
#' @seealso \code{\link{jackVar}}, 
#' \code{\link{jackFunc}},
#' \code{\link{regenboot}},
#' \code{\link{jackFuncRegen}}, 
#' \code{\link{jackFuncBlock}},
#' \code{\link{jackVarRegen.atom}},
#'  \code{\link{jackVarRegen.smallEnsemble}}.
#' @keywords Jackknife "Variance estimation" "Regenerative processes"
#' @export
#' @examples
#'  acgt <- c("A", "C", "G", "T")
#'  probs <- c(.3, .1, .3, .3)
#'  n <- 100
#'  atom <- "A"
#'  set.seed(1)
#'  y <- sample(acgt, n, prob=probs, repl=TRUE)
#'  propAtom <- function(x) {
#'    tbl <- as.vector(table(x))
#'    prop <- tbl[1] / length(x)
#'    return(prop)
#'  }
#'  jackVarRegen(y, propAtom, atom=atom)
##
jackVarRegen <- function(x,func,...,atom,small=NULL,s=median(x)) {
  if (!missing(atom)) {
    # This is the atomic case
    res <- jackVarRegen.atom(x,func,atom=atom,...)
  } else {
    if (is.null(small)) {
      small <- findBestEpsilon(x,s)
    }
    res <- jackVarRegen.smallEnsemble(x,func,small,...)
  }
  return(res)
}