## 
# ------------------------------------------------------------------------
# 
# "findBestEpsilon(x,s)" --
# 
# Find the best radius epsilon for the truncatures in small ensembles. The
# function returns a list with the following computed values:
#     * optimal epsilon
#     * corresponding lower bound delta
#     * estimated transition densities of Markov chain p_n(X_i,X_{i+1})
# 
# ------------------------------------------------------------------------
#' @aliases findBestEpsilon
#' @title Optimal Size of Small Sets
#' @description 
#' Determines the optimal radius \eqn{\epsilon} of small sets, which maximizes the 
#' expected number of regeneration blocks. 
#' Useful for identifying the parameters for splitting a Markov chain into almost
#' independent blocks.
#' @param x A numeric vector representing a Markov chain.
#' @param s A real number specifying the center of the small set. 
#' Default is the median of \code{x}.
#' @param plotIt Logical. If \code{TRUE} plots the estimated expected number of 
#' regenerations as a 
#' function of the radius \eqn{\epsilon}.
#' @details The \code{findBestEpsilon} calculates the optimal radius \eqn{\epsilon} and the 
#' corresponding lower bound \eqn{\delta} of the transition density for choosing the 
#' optimal small set of a Markov chain given the center \code{s} of a small set. 
#' It uses the \code{fastNadaraya} function to estimate
#' the transition density \eqn{p_n(X_i,X_{i+1})} taken at points \eqn{(X_i,X_{i+1})}.
#' 
#' @return 
#'  Returns an object of class \code{smallEnsemble}, which is a list containing:
#'   \itemize{
#'     \item \code{s}: the chosen center of the small set,
#'     \item \code{epsilon}: estimated optimal radius of the small set,
#'     \item \code{delta}: estimated lower bound of the transition density over the optimal 
#'     small set,
#'     \item \code{trans}: estimated values of \eqn{p_n(X_i,X_{i+1})}.
#'   }
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#' Bertail, P. and Clémençon, S. (2006). Regenerative block bootstrap for Markov
#'  chains. \emph{Bernoulli}, \bold{12}, 689-712. 
#' 
#' @seealso \code{\link{regenboot}}, \code{\link{fastNadaraya}}, \code{\link{naradamar}}, 
#' \code{\link{GetBlocks}}, \code{\link{GetPseudoBlocks}}, \code{\link{smallEnsemble}}.
#' @keywords "Regenerative Block Bootstrap" "Markov chains" "Optimal small set" 
#' @export
#' @examples
#' \donttest{
#' n=200 # the length of the process
#' # Generating the AR(1) process
#' sigma=1
#' coeff=0.75
#' X=arima.sim(n=n,list(ar=coeff, sd=sigma))
#' # Find the small ensemble with the largest number of regeneration
#' sm <- findBestEpsilon(X,s=0,plotIt=FALSE)
#' }
##
findBestEpsilon <- function(x,s=median(x),plotIt=FALSE) {
  n <- length(x)-1
  y <- x-s
  q <- quantile(abs(y),0.9)
  eps <- seq(from=0,q,len=200)[-1]
  leps <- length(eps)
  
  delta <- numeric(leps)
  N_eps <- numeric(leps)
  h <- bandw1(x)
  
  # Estimator of the transition densities p_n(X_i,X_{i+1})
  p_XiXip1 <- fastNadaraya(x,h)
  if (min(p_XiXip1) == 0) {
    loc0 <- (p_XiXip1 == 0)
    # To avoid NaN when dividing by p_XiXip1
    p_XiXip1[loc0] <- 1
    p_XiXip1[loc0] <- min(p_XiXip1)/10
  } 
  
  # Kernel estimator of the transition. S is a subdivision of the
  # interval [s-q,s+q] (q is max(eps)).
  S <- seq(from=s-q,to=s+q,len=500)
  f <- naradamar(S,S,x,h)
  
  # Compute the estimation of the number of blocks for all the values of
  # epsilon
  for (j in 1:leps) {
    e <- eps[j]
    l_eps <- sum(S <= s-e)
    u_eps <- sum(S <= s+e)
    delta[j] <- min(f[l_eps:u_eps,l_eps:u_eps])
    # Indicator if pairs (X_i,X_{i+1}) located inside the interval [s-epsilon,s+epsilon]
    voisin <- (x[1:n]>=s-e)*(x[2:(n+1)]>=s-e)*(x[1:n]<=s+e)*(x[2:(n+1)]<=s+e)
    # Estimator of the number of regenerative blocks
    N_eps[j]=delta[j]*sum(voisin/p_XiXip1)
  }
  
  optidx <- which.max(N_eps)
  
  if (plotIt) {
    plot(eps,N_eps,type="l",xlab="epsilon",ylab=expression(N[eps]))
    abline(h=0,col="gray")
    abline(v=eps[optidx],col="red")
    text(eps[optidx],1,round(eps[optidx],4))
  }
  res <- list(s=s,epsilon=eps[optidx],delta=delta[optidx],trans=p_XiXip1)
  class(res) <- "smallEnsemble"
  return(res)
}