
##
# ------------------------------------------------------------------------
#
# "boot_dist(boot_obj,comp,PLT,nn,recenter)" --
#
#
# ------------------------------------------------------------------------
##
#' @aliases boot_dist
#' @title Bootstrap Distribution
#' @description This function computes the rescaled and recentered bootstrap
#'  distribution from a given bootstrap object. It optionally plots this 
#'  distribution and superimposes a normal approximation for comparison.
#' @param boot_obj A bootstrap object of type \code{boodd}, containing the bootstrap samples.
#' @param comp A positive Integer; The index of the column of the bootstrap matrix to be plotted or analysed.
#' @param PLT Logical. If \code{TRUE} (default), the function plots the distribution.
#' @param nn Logical. If \code{TRUE} (default), a normal approximation is superimposed 
#'        on the histogram.
#' @param recenter Logical. If \code{TRUE}, the bootstrap distribution is recentered by
#'  the bootstrap mean, else if \code{FALSE} (default)
#' it is recentered by the original value of the estimator. 
#' @param return_values Logical. If \code{TRUE} the function returns the calculated values
#' for the rescaled and recentered distribution. \code{FALSE} by default.
#' @details The function modifies the bootstrap distribution by recentering it, 
#' depending on value of  \code{recenter} 
#' parameter and computes a standardized version of the selected component of the statistic.
#'  If \code{PLT} is \code{TRUE}, it generates a histogram of the 
#' standardized distribution with optional normal approximation overlay if \code{nn} 
#' is \code{TRUE}.
#' @returns  A numeric vector representing the rescaled and optionally recentered 
#'         bootstrap distribution of the selected component of the statistic.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted. 
#' @seealso \code{\link{plot.boodd}}.
#' @keywords Bootstrap Recentering Rescaling "Normal approximation"
#' @export
#' @examples
#' n=30
#' X=rnorm(n)
#' fun = function(x){
#' c(mean(x), var(x))}
#' boo=boots(X, fun, B=999)
#' # Recentered, rescaled bootstrap distribution of the first component - mean
#' boot_dist(boo, comp=1, recenter = TRUE) 
#' # Rescaled bootstrap distribution of the first component - variance
#' boot_dist(boo, comp=2) 


boot_dist=function(boot_obj,comp=1,PLT=TRUE,nn=TRUE,recenter=FALSE, return_values=FALSE){
  boot_obj$s=as.matrix(boot_obj$s)
  d=dim(boot_obj$s)
  var_star=var(boot_obj$s[,comp])
  
  if (recenter) {bb=(boot_obj$s[,comp]-mean(boot_obj$s[,comp]))/sqrt(var_star)}
  else  {bb=(boot_obj$s[,comp]-boot_obj$Tn[comp])/sqrt(var_star)}
  if (PLT==TRUE){
    hist(bb,nclass=35,freq=FALSE,main="Standardized bootstrap distribution")
    lines(density(bb),col="red",type="l")}
  if (nn==TRUE) curve(dnorm,from=-3,to=3,add=TRUE,col="green",type="l")
  if (return_values==TRUE)
    return(bb)
}