##
# ------------------------------------------------------------------------
#
# "blockboot.seasonal <- function(x,func,B,length.block,d,...,
#    method=c("movingblock","circular"),d )" --
#
# Generalized seasonal block bootstrap (GSBB) for time series.
#
# ------------------------------------------------------------------------
##
#' @aliases blockboot.seasonal
#' @title Generalized Seasonal Block Bootstrap for Time Series.
#' @description  
#' Applies generalized block bootstrap methods, including the Generalized Seasonal Block 
#' Bootstrap (GSBB) and the Circular Generalized Seasonal Block Bootstrap (CGSBB), to 
#' periodically correlated (PC) processes in time series analysis.
#' @param x A vector or a time series.
#' @param func A function to apply to each sample.
#' @param B A positive integer; the number of bootstrap replications.
#' @param length.block A positive integer; the length of the blocks; 
#'        if \code{NULL}, a default value is used based on the time series period.
#'        The default value is \code{d+1}.
#' @param method The bootstrap method to use; can be \code{"GSBB"} for Generalized Seasonal 
#'        Block Bootstrap or \code{"CGSBB"} for Circular Generalized Seasonal Block Bootstrap.
#' @param d The length of the seasonal period in the time series.
#' @param ... Optional additional arguments for the \code{func} function.
#' @details 
#' This function is designed to handle the bootstrapping of time series data 
#' that exhibit periodic correlation. It allows the use of two methods: 
#' GSBB, which is based on moving blocks, and CGSBB, which uses a circular 
#' approach. The function selects blocks of data with a specified length and 
#' applies the user-defined function to each bootstrap sample. The user may calibrate 
#' the optimal size of the blocks by using the \code{bopt_circy} function.
#'
#' The function adapts to the seasonal structure of the data, making it suitable 
#' for time series with inherent periodicity. The choice between GSBB and CGSBB 
#' methods can be made according to the specific characteristics of the time series.
#'
#' @return An object of class \code{boodd} containing the results of the bootstrap 
#'         procedure and the statistic computed by the \code{func} function on the original data.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#' Patrice Bertail, Anna E. Dudek. (2022). Optimal choice of bootstrap block 
#' length for periodically correlated time series. \emph{Bernoulli}, \bold{30}, 2521-2545.
#' 
#' Dudek, A.E., Leśkow, J., Paparoditis, E. and Politis, D. (2014a). 
#' A generalized block bootstrap for seasonal time series. \emph{J. Time Ser. Anal.}, 
#' \bold{35}, 89-114.
#' @seealso \code{\link{blockboot}}, \code{\link{bopt_circy}}.
#' @keywords "Bootstrap" "Periodically correlated" "Season"
#' @examples
#' set.seed(54321)
#' n <- 1000 # sample size
#'  f0 <- 1/12
#'  u <- rnorm(n, 0, 0.4)
#'  coeff=0.2
#'  b=rep(0,n)
#'  for (i in 2:(n-1)){ b[i+1]=coeff*b[i]+rnorm(1,0,0.4)}
#'  # Simulate a seasonal process with period 12
#'  X <- 5*cos(2 * pi * f0 * (1:n))+5*b * cos(2 * pi * f0 * (1:n)) + u
#'  X_ts <- ts(X)
#'  b=15 # Block length
#'  result = blockboot.seasonal(X_ts, mean, B=999, length.block=b, d=12, method="GSBB")
#'  plot(result, nclass=30)
#'  confint(result) 
#' @export
##
blockboot.seasonal <- function(x,func,B,length.block=NULL,...,method=c("GSBB","CGSBB"),d) {
  method <- match.arg(method)
  if (is.null(length.block)) {
    defaultlen <- d+1
  }
  
  if (length.block >= length(x)) {
    stop("the block length must be less than the size of the time series")
  }
  # Test the value returned by func
  y <- as.vector(func(x,...))
  #if (!is.vector(y)) {
  #  stop("Function 'func' must return a vector")
  #}
  len <- length(y)
  cnames <- names(y)
  
  if (method[1] == "GSBB") {
    res <- blockboot.seasonal.gsbb(x,func,B,length.block,d,...)
  } else {
    res <- blockboot.seasonal.cgsbb(x,func,B,length.block,d,...)
  }
  obj <- list(s=res,Tn=y)
  class(obj) <- "boodd"
  attr(obj,"kind") <- c("block",method[1])
  attr(obj,"func") <- func
  return(obj)
  return(res)
}


blockboot.seasonal.gsbb <- function(x,func,B,length.block,d,...) { 
  n <- length(x)
  y <- func(x,...)
  b <- length.block
  limit <- n-b+1
  w <- n%/%d
  # Allocate the result object
  len <- length(y)
  res <- matrix(nrow=B,ncol=len)
  # Number of sections
  s <- ((n-1) %/% b)+1
  # Bootstrapped series
  nx <- numeric(s*b)
  lp<-limit %/% d
  lr<-limit %% d
  base<-rep(lp,d)
  if (lr!= 0) {
    for(i in 1:lr){
    base[i] <- base[i]+1 # base[i] contains number of possible blocks of length b that start with an observation from season i
  }} 
  
  for (i in 1:B) {
    c <- 1
    pos <- 1
    while (pos <= n) {
      # Draw a number of period in which block will start
      sp <-sample.int(base[c],1)
      idx<-c+(sp-1)*d # index of the first observation in the selected block
      nx[pos:(pos+b-1)] <- x[idx:(idx+b-1)]
      pos <- pos+b
      c <- (c+b-1)%%d+1
    }
    res[i,] <- func(nx[1:n],...)
  }
  return(res)
}


blockboot.seasonal.cgsbb <- function(x,func,B,length.block,d,...) { 
  n <- length(x)
  y <- func(x,...)
  b <- length.block
  w <- n%/%d
  x <- x[1:(w*d)]
  # Ensure circularity
  x <- c(x,x[1:b])
  # Allocate the result object
  len <- length(y)
  res <- matrix(nrow=B,ncol=len)
  # Number of sections
  s <- ((n-1) %/% b)+1
  # Bootstrapped series
  nx <- numeric(s*b)
  
  for (i in 1:B) {
    c <- 1
    pos <- 1
    # The number of periodic indices c, c+d, c+2*d, ... up to n is w
    base <- (0:(w-1))*d
    while (pos <= n) {
      # Draw an index
      idx <- sample(c+base,1)
      nx[pos:(pos+b-1)] <- x[idx:(idx+b-1)]
      pos <- pos+b
      c <- (c+b-1)%%d+1
    }
    res[i,] <- func(nx[1:n],...)
  }
  return(res)
}