% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/site_species_metrics.R
\name{site_species_metrics}
\alias{site_species_metrics}
\title{Calculate contribution metrics of sites and species}
\usage{
site_species_metrics(
  bioregionalization,
  comat,
  indices = c("rho"),
  net = NULL,
  site_col = 1,
  species_col = 2
)
}
\arguments{
\item{bioregionalization}{A \code{bioregion.clusters} object.}

\item{comat}{A co-occurrence \code{matrix} with sites as rows and species as
columns.}

\item{indices}{A \code{character} specifying the contribution metric to compute.
Available options are \code{rho}, \code{affinity}, \code{fidelity}, \code{indicator_value} and
\code{Cz}.}

\item{net}{\code{NULL} by default. Required for \code{Cz} indices. A
\code{data.frame} where each row represents an interaction between two nodes
and an optional third column indicating the interaction's weight.}

\item{site_col}{A number indicating the position of the column containing
the sites in \code{net}. 1 by default.}

\item{species_col}{A number indicating the position of the column
containing the species in \code{net}. 2 by default.}
}
\value{
A \code{data.frame} with columns \code{Bioregion}, \code{Species}, and the desired summary
statistics, or a list of \code{data.frame}s if \code{Cz} and other indices are
selected.
}
\description{
This function calculates metrics to assess the contribution of a given
species or site to its bioregion.
}
\details{
The \eqn{\rho} metric is derived from Lenormand et al. (2019) with the
following formula:

\eqn{\rho_{ij} = \frac{n_{ij} - \frac{n_i n_j}{n}}{\sqrt{\left(\frac{n - n_j}{
n-1}\right) \left(1-\frac{n_j}{n}\right) \frac{n_i n_j}{n}}}}

where \eqn{n} is the number of sites, \eqn{n_i} is the number of sites in
which species \eqn{i} is present, \eqn{n_j} is the number of sites in
bioregion \eqn{j}, and \eqn{n_{ij}} is the number of occurrences of species
\eqn{i} in sites of bioregion \eqn{j}.

Affinity \eqn{A}, fidelity \eqn{F}, and individual contributions
\eqn{IndVal} describe how species are linked to their bioregions. These
metrics are described in Bernardo-Madrid et al. (2019):
\itemize{
\item Affinity of species to their region:
\eqn{A_i = \frac{R_i}{Z}}, where \eqn{R_i} is the occurrence/range size
of species \eqn{i} in its associated bioregion, and \eqn{Z} is the total
size (number of sites) of the bioregion. High affinity indicates that the
species occupies most sites in its bioregion.
\item Fidelity of species to their region:
\eqn{F_i = \frac{R_i}{D_i}}, where \eqn{R_i} is the occurrence/range size
of species \eqn{i} in its bioregion, and \eqn{D_i} is its total range size.
High fidelity indicates that the species is not present in other regions.
\item Indicator Value of species:
\eqn{IndVal = F_i \cdot A_i}.
}

\code{Cz} metrics are derived from Guimerà & Amaral (2005):
\itemize{
\item Participation coefficient:
\eqn{C_i = 1 - \sum_{s=1}^{N_M}{\left(\frac{k_{is}}{k_i}\right)^2}}, where
\eqn{k_{is}} is the number of links of node \eqn{i} to nodes in bioregion
\eqn{s}, and \eqn{k_i} is the total degree of node \eqn{i}. A high value
means links are uniformly distributed; a low value means links are within
the node's bioregion.
\item Within-bioregion degree z-score:
\eqn{z_i = \frac{k_i - \overline{k_{si}}}{\sigma_{k_{si}}}}, where
\eqn{k_i} is the number of links of node \eqn{i} to nodes in its bioregion
\eqn{s_i}, \eqn{\overline{k_{si}}} is the average degree of nodes in
\eqn{s_i}, and \eqn{\sigma_{k_{si}}} is the standard deviation of degrees
in \eqn{s_i}.
}
}
\examples{
comat <- matrix(sample(0:1000, size = 500, replace = TRUE, prob = 1/1:1001),
                20, 25)
rownames(comat) <- paste0("Site",1:20)
colnames(comat) <- paste0("Species",1:25)

dissim <- dissimilarity(comat, metric = "Simpson")
clust1 <- nhclu_kmeans(dissim, n_clust = 3, index = "Simpson")

net <- similarity(comat, metric = "Simpson")
com <- netclu_greedy(net)

site_species_metrics(bioregionalization = clust1, comat = comat,
indices = "rho")

# Contribution metrics
site_species_metrics(bioregionalization = com, comat = comat,
indices = c("rho", "affinity", "fidelity", "indicator_value"))

# Cz indices
net_bip <- mat_to_net(comat, weight = TRUE)
clust_bip <- netclu_greedy(net_bip, bipartite = TRUE)
site_species_metrics(bioregionalization = clust_bip, comat = comat, 
net = net_bip, indices = "Cz")

}
\references{
Bernardo-Madrid R, Calatayud J, González‐Suárez M, Rosvall M, Lucas P,
Antonelli A & Revilla E (2019) Human activity is altering the world’s
zoogeographical regions. \emph{Ecology Letters} 22, 1297--1305.

Guimerà R & Amaral LAN (2005) Functional cartography of complex metabolic
networks. \emph{Nature} 433, 895--900.

Lenormand M, Papuga G, Argagnon O, Soubeyrand M, Alleaume S & Luque S (2019)
Biogeographical network analysis of plant species distribution in the
Mediterranean region. \emph{Ecology and Evolution} 9, 237--250.
}
\seealso{
For more details illustrated with a practical example,
see the vignette:
\url{https://biorgeo.github.io/bioregion/articles/a5_3_summary_metrics.html}.

Associated functions:
\link{bioregion_metrics} \link{bioregionalization_metrics}
}
\author{
Pierre Denelle (\email{pierre.denelle@gmail.com}) \cr
Boris Leroy (\email{leroy.boris@gmail.com}) \cr
Maxime Lenormand (\email{maxime.lenormand@inrae.fr})
}
