% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/NSSHeat.R
\name{DissectMechanism}
\alias{DissectMechanism}
\title{Construct heatmap for non-steady state (NSS) analysis with improved mechanism score}
\usage{
DissectMechanism(
  bakRFit,
  DE_df,
  bakRModel = c("MLE", "Hybrid", "MCMC"),
  DE_cutoff = 0.05,
  bakR_cutoff = 0.3,
  Exp_ID = 2,
  sims = 1e+07
)
}
\arguments{
\item{bakRFit}{bakRFit object}

\item{DE_df}{dataframe of required format with differential expression analysis results. See
Further-Analyses vignette for details on what this dataframe should look like}

\item{bakRModel}{Model fit from which bakR implementation should be used? Options are MLE, Hybrid,
or MCMC}

\item{DE_cutoff}{padj cutoff for calling a gene differentially expressed}

\item{bakR_cutoff}{padj cutoff for calling a fraction new significantly changed. As discussed in the mechanistic
dissection vignette, it is best to keep this more conservative (higher padj) than is typical. Thus, default is 0.3 rather
than the more standard (though admittedly arbitrary) 0.05.}

\item{Exp_ID}{Exp_ID of experimental sample whose comparison to the reference sample you want to use.
Only one reference vs. experimental sample comparison can be used at a time}

\item{sims}{Number of simulation draws from null distribution for mechanism p value calculation}
}
\value{
returns list of data frames: heatmap_df and NSS_stats.
The heatmap_dfdata frame can be passed to pheatmap::pheatmap().
The NSS_stats data frame contains all of the information passed to NSS_stats as well
as the raw mechanism scores. It also has p values calculated from the mechanism z scores.
}
\description{
This uses the output of bakR and a differential expression analysis software to construct
a dataframe that can be passed to pheatmap::pheatmap(). This heatmap will display the
result of a steady-state quasi-independent analysis of NR-seq data.
}
\details{
Unlike NSSHeat, DissectMechanism uses a mechanism scoring function that is not discontinuous
as the "degradation driven" vs. "synthesis driven" boundary. Instead, the score
approaches 0 as the function approaches the boundary from either side.

In addition, DissectMechanism now defines a null model for the purpose of p value calculation using
the mechanism score. Under the null hypothesis, the mechanism score is the product of two
normal distributions with unit variance, one which has a non-zero mean. Simulation is used
to estimate the integral of this distribution, and the number of draws (which determines the
precision of the p value estimate) is determined by the \code{sims} parameter.

DissectMechanism also provides "meta-analysis p values", which can be interpreted as the p-value that
a particular RNA feature is observing differential expression or differential kinetics (or both).
This meta_pval is estimated using Fisher's method for meta analysis.
}
\examples{
\donttest{
# Simulate small dataset
sim <- Simulate_bakRData(100, nreps = 2)

# Analyze data with bakRFit
Fit <- bakRFit(sim$bakRData)

# Number of features that made it past filtering
NF <- nrow(Fit$Fast_Fit$Effects_df)

# Simulate mock differential expression data frame
DE_df <- data.frame(XF = as.character(1:NF),
                       L2FC_RNA = stats::rnorm(NF, 0, 2))

DE_df$DE_score <- DE_df$L2FC_RNA/0.5
DE_df$DE_se <- 0.5

DE_df$DE_pval <- 2*stats::dnorm(-abs(DE_df$DE_score))
DE_df$DE_padj <- 2*stats::p.adjust(DE_df$DE_pval, method = "BH")

# perform NSS analysis
NSS_analysis <- DissectMechanism(bakRFit = Fit,
               DE_df = DE_df,
               bakRModel = "MLE")

}
}
