% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/adversarial_rf.R
\name{adversarial_rf}
\alias{adversarial_rf}
\title{Adversarial Random Forests}
\usage{
adversarial_rf(
  x,
  num_trees = 10L,
  min_node_size = 2L,
  delta = 0,
  max_iters = 10L,
  early_stop = TRUE,
  prune = TRUE,
  verbose = TRUE,
  parallel = TRUE,
  ...
)
}
\arguments{
\item{x}{Input data. Integer variables are recoded as ordered factors with
a warning. See Details.}

\item{num_trees}{Number of trees to grow in each forest. The default works
well for most generative modeling tasks, but should be increased for
likelihood estimation. See Details.}

\item{min_node_size}{Minimal number of real data samples in leaf nodes.}

\item{delta}{Tolerance parameter. Algorithm converges when OOB accuracy is
< 0.5 + \code{delta}.}

\item{max_iters}{Maximum iterations for the adversarial loop.}

\item{early_stop}{Terminate loop if performance fails to improve from one
round to the next?}

\item{prune}{Impose \code{min_node_size} by pruning?}

\item{verbose}{Print discriminator accuracy after each round? Will also show
additional warnings.}

\item{parallel}{Compute in parallel? Must register backend beforehand, e.g.
via \code{doParallel} or \code{doFuture}; see examples.}

\item{...}{Extra parameters to be passed to \code{ranger}.}
}
\value{
A random forest object of class \code{ranger}.
}
\description{
Implements an adversarial random forest to learn independence-inducing splits.
}
\details{
The adversarial random forest (ARF) algorithm partitions data into fully
factorized leaves where features are jointly independent. ARFs are trained
iteratively, with alternating rounds of generation and discrimination. In
the first instance, synthetic data is generated via independent bootstraps of
each feature, and a RF classifier is trained to distinguish between real and
fake samples. In subsequent rounds, synthetic data is generated separately in
each leaf, using splits from the previous forest. This creates increasingly
realistic data that satisfies local independence by construction. The
algorithm converges when a RF cannot reliably distinguish between the two
classes, i.e. when OOB accuracy falls below 0.5 + \code{delta}.

ARFs are useful for several unsupervised learning tasks, such as density
estimation (see \code{\link{forde}}) and data synthesis (see
\code{\link{forge}}). For the former, we recommend increasing the number of
trees for improved performance (typically on the order of 100-1000 depending
on sample size).

Integer variables are recoded with a warning (set \code{verbose = FALSE} to
silence these). Default behavior is to convert integer variables with six or
more unique values to numeric, while those with up to five unique values are
treated as ordered factors. To override this behavior, explicitly recode
integer variables to the target type prior to training.

Note: convergence is not guaranteed in finite samples. The \code{max_iters}
argument sets an upper bound on the number of training rounds. Similar
results may be attained by increasing \code{delta}. Even a single round can
often give good performance, but data with strong or complex dependencies may
require more iterations. With the default \code{early_stop = TRUE}, the
adversarial loop terminates if performance does not improve from one round
to the next, in which case further training may be pointless.
}
\examples{
# Train ARF and estimate leaf parameters
arf <- adversarial_rf(iris)
psi <- forde(arf, iris)

# Generate 100 synthetic samples from the iris dataset
x_synth <- forge(psi, n_synth = 100)

# Condition on Species = "setosa" and Sepal.Length > 6
evi <- data.frame(Species = "setosa",
                  Sepal.Length = "(6, Inf)")
x_synth <- forge(psi, n_synth = 100, evidence = evi)

# Estimate average log-likelihood
ll <- lik(psi, iris, arf = arf, log = TRUE)
mean(ll)

# Expectation of Sepal.Length for class setosa
evi <- data.frame(Species = "setosa")
expct(psi, query = "Sepal.Length", evidence = evi)

\dontrun{
# Parallelization with doParallel
doParallel::registerDoParallel(cores = 4)

# ... or with doFuture
doFuture::registerDoFuture()
future::plan("multisession", workers = 4)
}

}
\references{
Watson, D., Blesch, K., Kapar, J., & Wright, M. (2023). Adversarial random
forests for density estimation and generative modeling. In \emph{Proceedings
of the 26th International Conference on Artificial Intelligence and
Statistics}, pp. 5357-5375.
}
\seealso{
\code{\link{arf}}, \code{\link{forde}}, \code{\link{forge}},
\code{\link{expct}}, \code{\link{lik}}
}
