% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/inference.R
\name{alphaci}
\alias{alphaci}
\alias{alphaci_std}
\title{Confidence intervals for alpha and standardized alpha}
\usage{
alphaci(
  x,
  type = c("adf", "elliptical", "normal"),
  transform = "none",
  parallel = FALSE,
  conf_level = 0.95,
  alternative = c("two.sided", "greater", "less"),
  bootstrap = FALSE,
  n_reps = 1000
)

alphaci_std(
  x,
  type = c("adf", "elliptical", "normal"),
  transform = "none",
  parallel = FALSE,
  conf_level = 0.95,
  alternative = c("two.sided", "greater", "less"),
  bootstrap = FALSE,
  n_reps = 1000
)
}
\arguments{
\item{x}{Input data data can be converted to a matrix using \code{as.matrix}.
Rows containing missing values are ignored.}

\item{type}{Type of confidence interval. Either \code{adf}, \code{elliptical}, or
\code{normal}.}

\item{transform}{One of \code{"none"}, \code{"log"}, \code{"fisher"}, and \verb{"arcsin}.
Defaults to \code{"none"}.}

\item{parallel}{If \code{TRUE}, makes calculations under the assumption of a
parallel model. Defaults to \code{FALSE}.}

\item{conf_level}{Confidence level. Defaults to \code{0.95}.}

\item{alternative}{A character string specifying the alternative hypothesis,
must be one of \code{"two.sided"} (default), \code{"greater"} or \code{"less"}.}

\item{bootstrap}{If \code{TRUE}, performs a studentized bootstrap with \code{n_reps}
repetitions. Defaults to \code{FALSE}.}

\item{n_reps}{Number of bootstrap samples if \code{bootstrap = TRUE}. Ignored if
\code{bootstrap = FALSE}. Defaults to \code{1000}.}
}
\value{
A vector of class \code{alphaci} containing the confidence end points.
The arguments of the function call are included as attributes.
}
\description{
Calculate confidence intervals for coefficient alpha (Cronbach, 1951)
and standardized alpha (Falk & Savalei, 2011) using asymptotic methods
or the studentized bootstrap. \code{alphaci} constructs confidence intervals
for coefficient alpha and \code{alphaci_std} for standardized alpha.
}
\details{
The methods accept handle missing data using \code{\link[stats:na.fail]{stats::na.omit}}, i.e., rows
containing missing data are removed. The bootstrap option uses the
studentized bootstrap (Efron, B. 1987), which is second order correct.
Both functions makes use of \code{\link{future.apply}} when bootstrapping.

The \code{type} variables defaults to \code{adf}, asymptotically distribution-free,
which is consistent when the fourth moment is finite
(Maydeu-Olivares et al. 2007). The \code{normal} option assumes normality.
(van Zyl et al. 1999), and is not concistent for models with excess
kurtosis unequal to \code{0}. The \code{elliptical} option assumes an
elliptical or pseudo-elliptical distribution of the data. The resulting
confidence intervals are corrected variants of the normal theory
intervals with a kurtosis correction (Yuan & Bentler 2002). The
common kurtosis parameter is calculated using the unbiased sample
kurtosis (Joanes, 1998). All these methods have analogues for
standardized alpha, which can be derived using the methods of Hayashi &
Kamata (2005) and Neudecker (2006).
}
\examples{
library("alphaci")
library("psychTools")
x <- bfi[, 1:5]
x[, 1] <- 7 - x[, 1] # Reverse-coded item.
alphaci(x)
alphaci_std(x)

# Calculate confidence intervals with other options.
library("lavaan")
x <- lavaan::HolzingerSwineford1939[1:20, 7:9]
results <- c(
  alphaci(x, type = "adf", parallel = FALSE),
  alphaci(x, type = "adf", parallel = TRUE),
  alphaci(x, type = "elliptical", parallel = FALSE),
  alphaci(x, type = "elliptical", parallel = TRUE),
  alphaci(x, type = "normal", parallel = FALSE),
  alphaci(x, type = "normal", parallel = TRUE)
)
}
\references{
Falk, C. F., & Savalei, V. (2011). The relationship between unstandardized and standardized alpha, true reliability, and the underlying measurement model. Journal of Personality Assessment, 93(5), 445-453. https://doi.org/10.1080/00223891.2011.594129

Cronbach, L. J. (1951). Coefficient alpha and the internal structure of tests. Psychometrika, 16(3), 297-334. https://doi.org/10.1007/BF02310555#'

Efron, B. (1987). Better Bootstrap Confidence Intervals. Journal of the American Statistical Association, 82(397), 171-185. https://doi.org/10.2307/2289144

Maydeu-Olivares, A., Coffman, D. L., & Hartmann, W. M. (2007). Asymptotically distribution-free (ADF) interval estimation of coefficient alpha. Psychological Methods, 12(2), 157-176. https://doi.org/10.1037/1082-989X.12.2.157

van Zyl, J. M., Neudecker, H., & Nel, D. G. (2000). On the distribution of the maximum likelihood estimator of Cronbach's alpha. Psychometrika, 65(3), 271-280. https://doi.org/10.1007/BF02296146

Yuan, K.-H., & Bentler, P. M. (2002). On robustness of the normal-theory based asymptotic distributions of three reliability coefficient estimates. Psychometrika, 67(2), 251-259. https://doi.org/10.1007/BF02294845

Joanes, D. N., & Gill, C. A. (1998). Comparing measures of sample skewness and kurtosis. Journal of the Royal Statistical Society: Series D (The Statistician), 47(1), 183-189. https://doi.org/10.1111/1467-9884.00122

Hayashi, K., & Kamata, A. (2005). A note on the estimator of the alpha coefficient for standardized variables under normality. Psychometrika, 70(3), 579-586. https://doi.org/10.1007/s11336-001-0888-1

Neudecker, H. (2006). On the Asymptotic Distribution of the Natural Estimator of Cronbach's Alpha with Standardised Variates under Nonnormality, Ellipticity and Normality. In P. Brown, S. Liu, & D. Sharma (Eds.), Contributions to Probability and Statistics: Applications and Challenges (pp. 167-171). World Scientific. https://doi.org/10.1142/9789812772466_0013
}
