% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/04-aghq.R
\name{marginal_laplace}
\alias{marginal_laplace}
\title{Marginal Laplace approximation}
\usage{
marginal_laplace(
  ff,
  k,
  startingvalue,
  transformation = default_transformation(),
  optresults = NULL,
  control = default_control_marglaplace(),
  ...
)
}
\arguments{
\item{ff}{A function list similar to that required by \code{aghq}. However, each
function now takes arguments \code{W} and \code{theta}. Explicitly, this is a
list containing elements:
\itemize{
\item{\code{fn}}{: function taking arguments \code{W} and \code{theta} and returning a numeric
value representing the log-joint posterior at \code{W,theta}}
\item{\code{gr}}{: function taking arguments \code{W} and \code{theta} and returning a numeric
vector representing the gradient with respect to \code{W} of the log-joint posterior at \code{W,theta}}
\item{\code{he}}{: function taking arguments \code{W} and \code{theta} and returning a numeric
matrix representing the hessian with respect to \code{W} of the log-joint posterior at \code{W,theta}}
}}

\item{k}{Integer, the number of quadrature points to use. I suggest at least 3. k = 1 corresponds to a Laplace
approximation.}

\item{startingvalue}{A list with elements \code{W} and \code{theta}, which are numeric
vectors to start the optimizations for each variable. If you're using this method
then the log-joint posterior should be concave and these optimizations should not be
sensitive to starting values.}

\item{transformation}{Optional. Do the quadrature for parameter \code{theta}, but
return summaries and plots for parameter \code{g(theta)}. This applies to the \code{theta}
parameters only, not the \code{W} parameters.
\code{transformation} is either: a) an \code{aghqtrans} object returned by \code{aghq::make_transformation},
or b) a list that will be passed to that function internally. See \code{?aghq::make_transformation} for details.}

\item{optresults}{Optional. A list of the results of the optimization of the log
posterior, formatted according to the output of \code{aghq::optimize_theta}. The
\code{aghq::aghq} function handles the optimization for you; passing this list
overrides this, and is useful for when you know your optimization is too difficult to be
handled by general-purpose software. See the software paper for several examples of this.
If you're unsure whether this option is needed for your problem then it probably is not.}

\item{control}{A list with elements
\itemize{
\item{\code{method}: }{optimization method to use for the \code{theta} optimization:
\itemize{
\item{'sparse_trust' (default): }{\code{trustOptim::trust.optim}}
\item{'sparse': }{\code{trust::trust}}
\item{'BFGS': }{\code{optim(...,method = "BFGS")}}
}
}
\item{\code{inner_method}: }{optimization method to use for the \code{W} optimization; same
options as for \code{method}}
}
Default \code{inner_method} is 'sparse_trust' and default \code{method} is 'BFGS'.}

\item{...}{Additional arguments to be passed to \code{ff$fn}, \code{ff$gr}, and \code{ff$he}.}
}
\value{
If \code{k > 1}, an object of class \code{marginallaplace}, which includes
the result of calling \code{aghq::aghq} on
the Laplace approximation of \code{(theta|Y)}, .... See software paper for full details.
If \code{k = 1}, an object of class \code{laplace} which is the result of calling
\code{aghq::laplace_approximation} on
the Laplace approximation of \code{(theta|Y)}.
}
\description{
Implement the marginal Laplace approximation of Tierney and Kadane (1986) for
finding the marginal posterior \code{(theta | Y)} from an unnormalized joint posterior
\code{(W,theta,Y)} where \code{W} is high dimensional and \code{theta} is low dimensional.
See the \code{AGHQ} software paper for a detailed example, or Stringer et. al. (2020).
}
\examples{
logfteta2d <- function(eta,y) {
  # eta is now (eta1,eta2)
  # y is now (y1,y2)
  n <- length(y)
  n1 <- ceiling(n/2)
  n2 <- floor(n/2)
  y1 <- y[1:n1]
  y2 <- y[(n1+1):(n1+n2)]
  eta1 <- eta[1]
  eta2 <- eta[2]
  sum(y1) * eta1 - (length(y1) + 1) * exp(eta1) - sum(lgamma(y1+1)) + eta1 +
    sum(y2) * eta2 - (length(y2) + 1) * exp(eta2) - sum(lgamma(y2+1)) + eta2
}
set.seed(84343124)
n1 <- 5
n2 <- 5
n <- n1+n2
y1 <- rpois(n1,5)
y2 <- rpois(n2,5)
objfunc2d <- function(x) logfteta2d(x,c(y1,y2))
objfunc2dmarg <- function(W,theta) objfunc2d(c(W,theta))
objfunc2dmarggr <- function(W,theta) {
  fn <- function(W) objfunc2dmarg(W,theta)
  numDeriv::grad(fn,W)
}
objfunc2dmarghe <- function(W,theta) {
  fn <- function(W) objfunc2dmarg(W,theta)
  numDeriv::hessian(fn,W)
}

funlist2dmarg <- list(
  fn = objfunc2dmarg,
  gr = objfunc2dmarggr,
  he = objfunc2dmarghe
)
}
\seealso{
Other quadrature: 
\code{\link{aghq}()},
\code{\link{get_hessian}()},
\code{\link{get_log_normconst}()},
\code{\link{get_mode}()},
\code{\link{get_nodesandweights}()},
\code{\link{get_numquadpoints}()},
\code{\link{get_opt_results}()},
\code{\link{get_param_dim}()},
\code{\link{laplace_approximation}()},
\code{\link{marginal_laplace_tmb}()},
\code{\link{nested_quadrature}()},
\code{\link{normalize_logpost}()},
\code{\link{optimize_theta}()},
\code{\link{plot.aghq}()},
\code{\link{print.aghqsummary}()},
\code{\link{print.aghq}()},
\code{\link{print.laplacesummary}()},
\code{\link{print.laplace}()},
\code{\link{print.marginallaplacesummary}()},
\code{\link{summary.aghq}()},
\code{\link{summary.laplace}()},
\code{\link{summary.marginallaplace}()}
}
\concept{quadrature}
