% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/surv_term.R
\name{surv_term}
\alias{surv_term}
\title{Creates Data for One Type of Survival Outcome}
\usage{
surv_term(formula, id, data, time_fixef, with_frailty = FALSE, delayed = NULL)
}
\arguments{
\item{formula}{a two-sided \code{\link{formula}} with the survival outcome
on the left-hand side and fixed effect covariates on the right-hand
side. The left-hand side needs to be a \code{\link[survival]{Surv}} object and can
be either right-censored and left-truncated.}

\item{id}{the variable for the id of each individual.}

\item{data}{\code{\link{data.frame}} with at least the time variable.}

\item{time_fixef}{the time-varying fixed effects. See .e.g.
\code{\link{poly_term}}. This is for the baseline hazard. Note that many
basis expansions have boundary knots. It is important that these are set
to cover the full range of survival times including time zero for some
expansions.}

\item{with_frailty}{\code{TRUE} if there should be a frailty term.}

\item{delayed}{a vector with an entry which is \code{TRUE} if the
left-truncation time from the survival outcome is from a delayed entry.}
}
\value{
An object of class \code{surv_term} with data required for survival outcome.
}
\description{
Creates Data for One Type of Survival Outcome
}
\details{
The \code{time_fixef} should likely not include an intercept as this is
often included in \code{formula}.

The \code{delayed} argument is to account for delayed entry with terminal
events when observations are sampled in a way such that they must not have
had the event prior to their left-truncation time. In this case, the proper
complete data likelihood is

\deqn{\frac{a(u)h(t_{ij}\mid u)^{d_{ij}}S(t_{ij} \mid u)g(u)}{\int a(u) S(v_{ij} \mid u) du}}

and not

\deqn{a(u)h(t_{ij} \mid u)^{d_{ij}}\frac{S(t_{ij} \mid u)}{S(v_{ij} \mid u)}g(u)}

where \eqn{h} is conditional hazard, \eqn{S} is the conditional survival
function, \eqn{g} is additional conditional likelihood factors from other
outcomes, \eqn{a} is the random effect distribution, \eqn{t_{ij}} is the
observed time, \eqn{d_{ij}} is an event indicator, and \eqn{v_{ij}} is the
left truncation time.

The denominator in the proper complete likelihood becomes the expectation
over all delayed entries when a cluster has more than one delayed entry. See
van den Berg and Drepper (2016) and Crowther et al. (2016) for further
details.
}
\examples{
# load in the data
library(survival)
data(pbc, package = "survival")

# re-scale by year
pbcseq <- transform(pbcseq, day_use = day / 365.25)
pbc <- transform(pbc, time_use = time / 365.25)

# base knots on observed event times
bs_term_knots <-
  with(pbc, quantile(time_use[status == 2], probs = seq(0, 1, by = .2)))

boundary <- c(bs_term_knots[ c(1, length(bs_term_knots))])
interior <- c(bs_term_knots[-c(1, length(bs_term_knots))])

# create the survival term
s_term <- surv_term(
  Surv(time_use, status == 2) ~ 1, id = id, data = pbc,
  time_fixef = bs_term(time_use, Boundary.knots = boundary, knots = interior))
}
\references{
Crowther MJ, Andersson TM, Lambert PC, Abrams KR & Humphreys K (2016).
\emph{Joint
modelling of longitudinal and survival data: incorporating delayed entry and
an assessment of model misspecification}. Stat Med,
35(7):1193-1209. doi:10.1002/sim.6779

van den Berg GJ & Drepper B (2016). \emph{Inference for Shared-Frailty
Survival Models with Left-Truncated Data}. Econometric Reviews, 35:6,
1075-1098, doi: 10.1080/07474938.2014.975640
}
