% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MaximizeParsimony.R
\encoding{UTF-8}
\name{MaximizeParsimony}
\alias{MaximizeParsimony}
\alias{Resample}
\alias{EasyTrees}
\alias{EasyTreesy}
\title{Find most parsimonious trees}
\usage{
MaximizeParsimony(
  dataset,
  tree,
  ratchIter = 7L,
  tbrIter = 2L,
  startIter = 2L,
  finalIter = 1L,
  maxHits = NTip(dataset) * 1.8,
  maxTime = 60,
  quickHits = 1/3,
  concavity = Inf,
  ratchEW = TRUE,
  tolerance = sqrt(.Machine[["double.eps"]]),
  constraint,
  verbosity = 3L
)

Resample(
  dataset,
  tree,
  method = "jack",
  proportion = 2/3,
  ratchIter = 1L,
  tbrIter = 8L,
  finalIter = 3L,
  maxHits = 12L,
  concavity = Inf,
  tolerance = sqrt(.Machine[["double.eps"]]),
  constraint,
  verbosity = 2L,
  ...
)

EasyTrees()

EasyTreesy()
}
\arguments{
\item{dataset}{A phylogenetic data matrix of \pkg{phangorn} class
\code{phyDat}, whose names correspond to the labels of any accompanying tree.
Perhaps load into R using \code{\link[TreeTools]{ReadAsPhyDat}}.
Additive (ordered) characters can be handled using
\code{\link[TreeTools]{Decompose}}.}

\item{tree}{(optional) A bifurcating tree of class \code{\link[ape]{phylo}},
containing only the tips listed in \code{dataset}, from which the search
should begin.
If unspecified, an \link[=AdditionTree]{addition tree} will be generated from
\code{dataset}, respecting any supplied \code{constraint}.
Edge lengths are not supported and will be deleted.}

\item{ratchIter}{Numeric specifying number of iterations of the
parsimony ratchet \insertCite{Nixon1999}{TreeSearch} to conduct.}

\item{tbrIter}{Numeric specifying the maximum number of \acronym{TBR}
break points on a given tree to evaluate before terminating the search.
One "iteration" comprises selecting a branch to break, and evaluating
each possible reconnection point in turn until a new tree improves the
score. If a better score is found, then the counter is reset to zero,
and tree search continues from the improved tree.}

\item{startIter}{Numeric: an initial round of tree search with
\code{startIter} × \code{tbrIter} \acronym{TBR} break points is conducted in
order to locate a local optimum before beginning ratchet searches.}

\item{finalIter}{Numeric: a final round of tree search will evaluate
\code{finalIter} × \code{tbrIter} \acronym{TBR} break points, in order to
sample the final optimal neighbourhood more intensely.}

\item{maxHits}{Numeric specifying the maximum times that an optimal
parsimony score may be hit before concluding a ratchet iteration or final
search concluded.}

\item{maxTime}{Numeric: after \code{maxTime} minutes, stop tree search at the
next opportunity.}

\item{quickHits}{Numeric: iterations on subsampled datasets
will retain \code{quickHits} × \code{maxHits} trees with the best score.}

\item{concavity}{Determines the degree to which extra steps beyond the first
are penalized.  Specify a numeric value to use implied weighting
\insertCite{Goloboff1993}{TreeSearch}; \code{concavity} specifies \emph{k} in
\emph{k} / \emph{e} + \emph{k}. A value of 10 is recommended;
TNT sets a default of 3, but this is too low in some circumstances
\insertCite{Goloboff2018,Smith2019}{TreeSearch}.
Better still explore the sensitivity of results under a range of
concavity values, e.g. \code{k = 2 ^ (1:7)}.
Specify \code{Inf} to weight each additional step equally,
(which underperforms step weighting approaches
\insertCite{Goloboff2008,Goloboff2018,Goloboff2019,Smith2019}{TreeSearch}).
Specify \code{"profile"} to employ an approximation of profile parsimony
\insertCite{Faith2001}{TreeSearch}.}

\item{ratchEW}{Logical specifying whether to use equal weighting during
ratchet iterations, improving search speed whilst still facilitating
escape from local optima.}

\item{tolerance}{Numeric specifying degree of suboptimality to tolerate
before rejecting a tree.  The default, \code{sqrt(.Machine$double.eps)}, retains
trees that may be equally parsimonious but for rounding errors.
Setting to larger values will include trees suboptimal by up to \code{tolerance}
in search results, which may improve the accuracy of the consensus tree
(at the expense of resolution) \insertCite{Smith2019}{TreeSearch}.}

\item{constraint}{Either an object of class \code{phyDat}, in which case
returned trees will be perfectly compatible with each character in
\code{constraint}; or a tree of class \code{phylo}, all of whose nodes will occur
in any output tree.
See \code{\link[TreeTools:ImposeConstraint]{ImposeConstraint()}} and
\href{https://ms609.github.io/TreeSearch/articles/tree-search.html}{vignette}
for further examples.}

\item{verbosity}{Integer specifying level of messaging; higher values give
more detailed commentary on search progress. Set to \code{0} to run silently.}

\item{method}{Unambiguous abbreviation of \code{jackknife} or \code{bootstrap}
specifying how to resample characters.  Note that jackknife is considered
to give more meaningful results.}

\item{proportion}{Numeric between 0 and 1 specifying what proportion of
characters to retain under jackknife resampling.}

\item{\dots}{Additional parameters to \code{MaximizeParsimony()}.}
}
\value{
\code{MaximizeParsimony()} returns a list of trees with class
\code{multiPhylo}. This lists all trees found during each search step that
are within \code{tolerance} of the optimal score, listed in the sequence that
they were first visited, and named according to the step in which they were
first found; it may contain more than \code{maxHits} elements.
Note that the default search parameters may need to be increased in order for
these trees to be the globally optimal trees; examine the messages printed
during tree search to evaluate whether the optimal score has stabilized.

The return value has the attribute \code{firstHit}, a named integer vector listing
the number of optimal trees visited for the first time in each stage of
the tree search. Stages are named:
\itemize{
\item \code{seed}: starting trees;
\item \code{start}: Initial TBR search;
\item \code{ratchN}: Ratchet iteration \code{N};
\item \code{final}: Final TBR search.
The first tree hit for the first time in ratchet iteration three is named
\code{ratch3_1}.
}

\code{Resample()} returns a \code{multiPhylo} object containing a list of
trees obtained by tree search using a resampled version of \code{dataset}.
}
\description{
Search for most parsimonious trees using the parsimony ratchet and
\acronym{TBR} rearrangements, treating inapplicable data as such using the
algorithm of \insertCite{Brazeau2019;textual}{TreeSearch}.

Tree search will be conducted from a specified or automatically-generated
starting tree in order to find a tree with an optimal parsimony score,
under implied or equal weights, treating inapplicable characters as such
in order to avoid the artefacts of the standard Fitch algorithm
\insertCite{@see @Maddison1993; @Brazeau2019}{TreeSearch}.
Tree length is calculated using the MorphyLib C library
\insertCite{Brazeau2017}{TreeSearch}.
}
\details{
Tree search commences with \code{ratchIter} iterations of the parsimony ratchet
\insertCite{Nixon1999}{TreeSearch}, which bootstraps the input dataset
in order to escape local optima.
A final round of tree bisection and reconnection (\acronym{TBR})
is conducted to broaden the sampling of trees.

This function can be called using the R command line / terminal, or through
the "shiny" graphical user interface app (type \code{EasyTrees()} to launch).

The optimal strategy for tree search depends in part on how close to optimal
the starting tree is, the size of the search space (which increases
super-exponentially with the number of leaves), and the complexity of the
search space (e.g. the existence of multiple local optima).

One possible approach is to employ four phases:
\enumerate{
\item Rapid search for local optimum: tree score is typically easy to improve
early in a search, because the initial tree is often far from optimal.
When many moves are likely to be accepted, running several rounds of search
with a low value of \code{maxHits} and a high value of \code{tbrIter} allows many
trees to be evaluated quickly, hopefully moving quickly to a more promising
region of tree space.
\item Identification of local optimum:
Once close to a local optimum, a more extensive search
with a higher value of \code{maxHits} allows a region to be explored in more
detail.  Setting a high value of \code{tbrIter} will search a local
neighbourhood more completely
\item Search for nearby peaks:
Ratchet iterations allow escape from local optima.
Setting \code{ratchIter} to a high value searches the wider neighbourhood more
extensively for other nearby peaks; \code{ratchEW = TRUE} accelerates these
exploratory searches.  Ratchet iterations can be ineffective when \code{maxHits}
is too low for the search to escape its initial location.
\item Extensive search of final optimum.  As with step 2, it may be valuable to
fully explore the optimum that is found after ratchet searches to be sure
that the locally optimal score has been obtained.  Setting a high value of
\code{finalIter} performs a thorough search that can give confidence that further
searches would not find better (local) trees.
}

A search is unlikely to have found a global optimum if:
\itemize{
\item Tree score continues to improve on the final iteration.  If a local optimum
has not yet been reached, it is unlikely that a global optimum has
been reached.
Try increasing \code{maxHits}.
\item Successive ratchet iterations continue to improve tree scores.
If a recent ratchet iteration improved the score, rather than finding
a different region of tree space with the same optimal score, it is likely
that still better global optima remain to be found.  Try increasing
\code{ratchIter} (more iterations give more chance for improvement) and
\code{maxHits} (to get closer to the local optimum after each ratchet iteration).
\item Optimal areas of tree space are only visited by a single ratchet iteration.
(See vignette: \href{https://ms609.github.io/TreeSearch/articles/tree-space.html}{Exploring tree space}.)
If some areas of tree space are only found by one ratchet iteration, there
may well be other, better areas that have not yet been visited.
Try increasing \code{ratchIter}.
}

When continuing a tree search, it is usually best to start from an optimal
tree found during the previous iteration - there is no need to start from
scratch.

A more time consuming way of checking that a global optimum has been reached
is to repeat a search with the same parameters multiple times, starting
from a different, entirely random tree each time. If all searches obtain the
same optimal tree score despite their different starting points,
this score is likely to correspond to the global optimum.

For detailed documentation of the "TreeSearch" package, including full
instructions for loading phylogenetic data into R and initiating and
configuring tree search, see the
\href{https://ms609.github.io/TreeSearch/}{package documentation}.
}
\section{Resampling}{

Note that bootstrap support is a measure of the amount of data supporting
a split, rather than the amount of confidence that should be afforded the
grouping.
"Bootstrap support of 100\% is not enough, the tree must also be correct"
\insertCite{Phillips2004}{TreeSearch}.
See discussion in \insertCite{Egan2006;textual}{TreeSearch};
\insertCite{Wagele2009;textual}{TreeSearch};
\insertCite{Simmons2011}{TreeSearch};
\insertCite{Kumar2012;textual}{TreeSearch}.

For a discussion of suitable search parameters in resampling estimates, see
\insertCite{Muller2005;textual}{TreeSearch}.
The user should decide whether to start each resampling
from the optimal tree (which may be quicker, but result in overestimated
support values as searches get stuck in local optima close to the
optimal tree) or a random tree (which may take longer as more rearrangements
are necessary to find an optimal tree on each iteration).

For other ways to estimate clade concordance, see \code{\link[=SiteConcordance]{SiteConcordance()}}.
}

\examples{
## Only run examples in interactive R sessions
if (interactive()) {
  # launch "shiny" point-and-click interface
  EasyTrees()
  
  # Here too, use the "continue search" function to ensure that tree score
  # has stabilized and a global optimum has been found
}


# Load data for analysis in R
library("TreeTools")
data("inapplicable.phyData", package = "TreeSearch")
dataset <- inapplicable.phyData[["Asher2005"]]

# A very quick run for demonstration purposes
trees <- MaximizeParsimony(dataset, ratchIter = 0, startIter = 0,
                           tbrIter = 1, maxHits = 4, maxTime = 1/100,
                           concavity = 10, verbosity = 4)
names(trees)
cons <- Consensus(trees)

# In actual use, be sure to check that the score has converged on a global
# optimum, conducting additional iterations and runs as necessary.
 
if (interactive()) {
# Jackknife resampling
nReplicates <- 10
jackTrees <- replicate(nReplicates,
  #c() ensures that each replicate returns a list of trees
  c(Resample(dataset, trees, ratchIter = 0, tbrIter = 2, startIter = 1,
             maxHits = 5, maxTime = 1 / 10,
             concavity = 10, verbosity = 0))
 )

# In a serious analysis, more replicates would be conducted, and each
# search would undergo more iterations.

# Now we must decide what to do with the multiple optimal trees from
# each replicate.

# Set graphical parameters for plotting
oPar <- par(mar = rep(0, 4), cex = 0.9)

# Take the strict consensus of all trees for each replicate
# (May underestimate support)
JackLabels(cons, lapply(jackTrees, ape::consensus))

# Take a single tree from each replicate (here, the first)
# Potentially problematic if chosen tree is not representative
JackLabels(cons, lapply(jackTrees, `[[`, 1))

# Count iteration as support if all most parsimonious trees support a split;
# as contradiction if all trees contradict it; don't include replicates where
# not all trees agree on the resolution of a split.
labels <- JackLabels(cons, jackTrees)

# How many iterations were decisive for each node?
attr(labels, "decisive")

# Show as proportion of decisive iterations
JackLabels(cons, jackTrees, showFrac = TRUE)

# Restore graphical parameters
par(oPar)
}

# Tree search with a constraint
constraint <- MatrixToPhyDat(c(a = 1, b = 1, c = 0, d = 0, e = 0, f = 0))
characters <- MatrixToPhyDat(matrix(
  c(0, 1, 1, 1, 0, 0,
    1, 1, 1, 0, 0, 0), ncol = 2,
  dimnames = list(letters[1:6], NULL)))
MaximizeParsimony(characters, constraint = constraint, verbosity = 0)

}
\references{
\insertAllCited{}
}
\seealso{
Tree search \emph{via} graphical user interface: \code{\link[=EasyTrees]{EasyTrees()}}

Other split support functions: 
\code{\link{JackLabels}()},
\code{\link{Jackknife}()},
\code{\link{PresCont}()},
\code{\link{SiteConcordance}}
}
\author{
\href{https://smithlabdurham.github.io/}{Martin R. Smith}
(\href{mailto:martin.smith@durham.ac.uk}{martin.smith@durham.ac.uk})
}
\concept{split support functions}
