#' @title Superadditive check
#' @description This function checks if the given game is superadditive.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @param instance A logical value. By default, \code{instance=FALSE}.
#' @return \code{TRUE} if the game is superadditive, \code{FALSE} otherwise. If \code{instance=TRUE} and the game is not superadditive, the function also returns the positions (binary order positions if \code{binary=TRUE}; lexicographic order positions otherwise) of a pair of coalitions violating superadditivity.
#' @details A game \eqn{v\in G^N} is superadditive if \eqn{v(S \cup T) \ge v(S)+v(T)} for all \eqn{S,T \in 2^N} with \eqn{S \cap T = \emptyset}.
#'
#' A game \eqn{v\in G^N} is subadditive if \eqn{-v} is superadditive.
#' @examples
#' v <- c(2, 2, 4, 2, 4, 5, 6)
#' superadditivecheck(v, binary = TRUE, instance = TRUE)
#'
#' # How to check if a game is subadditive:
#' v.sub <- c(40, 30, 50, 60, 70, 65, 90) # subadditive game
#' superadditivecheck(-v.sub)
#' @seealso \link{additivecheck}, \link{convexcheck}, \link{monotoniccheck}, \link{strategicallyequivalentcheck}
#' @export
superadditivecheck <- function(v, binary = FALSE, instance = FALSE) { # Principio de la función

  ################################
  ### Comprobación datos entrada###
  ################################

  nC <- length(v) # Número de coaliciones.
  n <- log(nC + 1) / log(2) # Número de jugadores.
  if (n > floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }
  if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, lo pasamos a binario.
    v <- lex2bin(v)
  }

  ################################
  ##### Creación de variables######
  ################################

  checkR <- TRUE # Por defecto, mientras no encuentre un contraejemplo, checkR=TRUE
  exampleR <- NULL # Por defecto, no hay contraejemplo
  S <- 1 # Inicializo coalición S.
  tol <- 100*.Machine$double.eps # Nivel de precisión

  while (checkR & S <= (nC - 1) / 2) # Mientras no haya encontrado un contraejemplo y no haya superado el número máximo de coaliciones, hago la comprobación.
  { # Solo necesito ir hasta la mitad de las coaliciones. Si seguimos se repetirían las operaciones.
    R <- S + 1 # Inicializo coalición R. Solo compruebo con las coaliciones que vienen después de S.
    while (checkR & R <= (nC - 1)) {
      if (bitwAnd(R, S) == 0) # Compruebo si R y S son disjuntas.
        {
          if (v[R + S] - (v[R] + v[S]) + tol < 0) # v(S U R)<v(S)+v(R).
            {
              checkR <- FALSE
              exampleR <- c(S, R)
            }
        }
      R <- R + 1
    }
    S <- S + 1
  }

  ################################
  ###### Salidas de la función#####
  ################################

  if (instance == FALSE) { # Si el usuario sólo pide una salida, es decir, instance=FALSE:
    return(check = checkR)
  } else { # Si instance=TRUE, la función devuelve también las posiciones de las coaliciones S y R.
    if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, devolvemos las posiciones lexicográficas de un par de coaliciones que falla.
      return(list(check = checkR, example = c(codebin2lex(n, exampleR[1]), codebin2lex(n, exampleR[2]))))
    } else { # Si el juego se introdujo en binario, devolvemos las posiciones binarias de un par de coaliciones que falla.
      return(list(check = checkR, example = exampleR))
    }
  }
} # Fin de la función
