#' @title Sequencing game
#' @description Given a sequencing situation with an initial order, this function returns the characteristic function of the associated sequencing game.
#' @param p A vector containing the processing time of each job.
#' @param alpha A vector containing the cost per unit of time that each job generates while unfinished.
#' @param pi0 An initial order, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}.
#' @return The characteristic function of the sequencing game (interpreted as a savings game), as a vector in binary order if \code{binary=TRUE} and in lexicographic order otherwise.
#' @details Given a coalition \eqn{S\in 2^N}, \eqn{\Pi(S)} is the set of orders of \eqn{S}, that is, the set of all bijective functions from \eqn{S} to \eqn{\{1,\ldots, s\}}. A generic order of \eqn{S} is denoted by \eqn{\pi_S\in \Pi(S)}.
#' Given \eqn{i\in S} and \eqn{\pi_S\in \Pi(S)}, let \eqn{Pre^{\pi}(i) =\{j\in S : \pi_S(j) < \pi_S(i)\}} be the set of predecessors of \eqn{i} according to order \eqn{\pi_S}.
#'
#' A sequencing situation is a triple \eqn{(N,p,\alpha)} and, possibly, some (information on the) initial order, where \eqn{N=\{1,\ldots,n\}} is a finite set of agents, each one having one job that has to be processed on a machine. To simplify, agent \eqn{i}'s job is identified with \eqn{i}. The processing times
#' of the jobs are given by \eqn{p=(p_i)_{i\in N}} with \eqn{p_i> 0} for all \eqn{i\in N}. For each agent \eqn{i\in N} there is a cost function \eqn{c_i:(0,\infty)\rightarrow \mathbb{R}}, so that \eqn{c_i(t)} represents the cost incurred when job \eqn{i} is completed exactly at time \eqn{t}. Assuming that \eqn{c_i} is linear for all \eqn{i\in N}, there exist \eqn{\alpha_i,\beta_i\geq 0} such that \eqn{c_i(t)=\beta_i + \alpha_i t} for all \eqn{i\in N}, where \eqn{\beta_i} is a fixed service cost and \eqn{\alpha_i t} is the completion cost.
#'
#' For any \eqn{\pi\in \Pi(N)}, \eqn{C(S,\pi)} is the aggregate completion cost of coalition \eqn{S} in the order \eqn{\pi}, formally defined as \deqn{C(S,\pi)=\sum_{i\in S}\alpha_i\Big(p_i+\sum_{j\in Pre^{\pi}(i)}p_j\Big).}
#'
#' A sequencing situation with initial order is a quadruple \eqn{(N,p,\alpha,\pi_0)} where \eqn{\pi_0\in\Pi(N)} is the initial order of the jobs.
#'
#' A coalition \eqn{S\in 2^N} is said to be connected in order \eqn{\pi} if, for all \eqn{i, j\in S} and \eqn{k\in N}, \eqn{\pi(i) < \pi(k)< \pi(j)} implies \eqn{k\in S}. We say that a coalition \eqn{S'} is a component of \eqn{S} if \eqn{S'\subset S}, \eqn{S'} is connected, and for every \eqn{i\in S\backslash S'}, \eqn{S'\cup i} is not connected. The components of \eqn{S} form a partition of \eqn{S} that is denoted by \eqn{S/\pi_0}. Curiel et al. (1989) define the gain of swapping \eqn{i} and \eqn{j} as \eqn{g_{ij}=\max\{0,\alpha_jp_i-\alpha_ip_j\}}.
#'
#' The sequencing game \eqn{(N,v_{\pi_0})} is defined, for all \eqn{S\in 2^N}, by
#'
#' \deqn{v_{\pi_0}(S)=\sum_{S'\in S/\pi_0} \left(\sum_{i,j\in S':\pi_0(i)<\pi_0(j)}g_{ij} \right).}
#' @seealso \link{tailgame}
#' @examples
#' p <- c(1,2,3,4)
#' alpha <- c(4,5,1,2)
#' pi0 <- c(2,3,1,4)
#' sequencinggame(p, alpha, pi0)
#' @references Curiel, I., Pederzoli, G., & Tijs, S. (1989). Sequencing games. \emph{European Journal of Operational Research}, 40(3), 344-351.
#' @export

sequencinggame=function(p,alpha,pi0,binary=FALSE){ # Principio de la función

  ################################
  ### Comprobación datos entrada##
  ################################

  n=length(p)#Número de trabajos.
  nC=(2^n)-1 #Número de coaliciones.
  if(length(alpha)!=n){
    stop("'p', 'alpha' and 'pi0' must have the same dimension.")
  }

  if(length(pi0)!=n){
    stop("'p', 'alpha' and 'pi0' must have the same dimension.")
  }

  if(sum(p<=0)>0){
    stop("All the elements of 'p' must be positive.")
  }

  if(sum(alpha<=0)>0){
    stop("All the elements of 'alpha' must be positive.")
  }

  if(sum(pi0<=0)>0){
    stop("All the elements of 'pi0' must be positive.")
  }

  if(sum(pi0!=unique(pi0))>0){
    stop("There is at least one job that appears in 'pi0' more than once.")
  }

  ################################
  #####Creación de variables######
  ################################

  u=alpha/p #Urgencia
  jug=seq(1:n)
  Optimal=jug[order(-u[jug])] #Orden óptimo
  v=rep(0,nC) #Iniciamos el juego.

  ################################
  ######Cuerpo de la función######
  ################################

  for(S in 1:(nC))
  {
    piS=c(rep(0,n))#Creamos un order artificial para cada S.
    coalition=getcoalition(S)
    if(length(coalition)>1)
    {
      for(j in 1:n)
      {
        for(k in coalition)
        {
          if(j==k)
          {#Cogemos el orden inicial y asignamos un cero a los elementos que no
            #están en la coalición S.
            piS[which(pi0==j)]=k #0 si el jugador no está en la coalición.
          }
        }
      }
    }
    for(j in 1:(n-1))
    {
      if(piS[j]!=0)
      {#Vamos viendo, uno a uno, los elementos conectados.
        k=j+1
        while(k<=n & piS[k]!=0)
        {
          if(u[piS[j]]<u[piS[k]])
          {#Si al cambiarlos, reducimos el coste, asignamos el cambio a la coalición S.
            v[S]=v[S]+alpha[piS[k]]*p[piS[j]]-alpha[piS[j]]*p[piS[k]]
          }
          k=k+1
        }
      }
    }
  }

  ################################
  ######Salidas de la función#####
  ################################
  if (binary == FALSE) { # Devolvemos el juego en orden lexicográfico.
    return(bin2lex(v))
  } else { # Devolvemos el juego en orden binario.
    return(v)
  }
}# Fin de la función
