#' @title Prenucleolus
#' @description Given a game, this function computes its prenucleolus.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @param tol A tolerance parameter, as a non-negative number.\cr
#'            By default, \code{tol=100*.Machine$double.eps}.
#' @return The prenucleolus of the game, as a vector.
#' @details Given a game \eqn{v\in G^N} and an allocation \eqn{x}, the excess of coalition \eqn{S \in 2^N}
#' with respect to \eqn{x} is defined as \eqn{e(v,x,S)=v(S)-x(S)}, where \eqn{x(S)=\sum_{i\in S} x_i}.
#' Let \eqn{\theta(x)} be a vector of excesses at \eqn{x} arranged in non-increasing order.
#' It is said that a vector \eqn{\alpha} is lexicographically greater than another vector \eqn{\beta}
#' if \eqn{\alpha \neq \beta} and the first non-zero coordinate of vector \eqn{\alpha-\beta} is positive.
#'
#' The prenucleolus is the set of the efficient allocations that produce a lexicographically minimal vector of excesses.
#' It is always non-empty and it actually comprises a single allocation,
#' which in zero-monotonic games coincides with the nucleolus.
#'
#' The implementation of this function is based on the algorithm presented in Derks and Kuipers (1997) and on the MATLAB package WCGT2005 by J. Derks.
#' @examples
#' prenucleolusvalue(c(0,0,0,0,10,40,30,60,10,20,90,90,90,130,160))
#' v <- runif(2^6-1, min = 10, max = 20) # random 6-player game
#' prenucleolusvalue(v)
#'
#' # The prenucleolus of v must pass the Kohlberg criterion.
#' # In some cases, though, the tolerance might have to be adjusted
#' # to avoid numerical error:
#' kohlbergcriterion(v,prenucleolusvalue(v))
#' kohlbergcriterion(v,prenucleolusvalue(v),tol=10^(-6))
#'
#' # What if the game is a cost game?
#' cost.v <- c(2,2,2,3,4,4,5) # cost game
#' -prenucleolusvalue(-cost.v) # prenucleolus of the cost game
#' @references Derks, J. & Kuipers, J. (1997). Implementing the simplex method for computing the prenucleolus of transferable utility games.
#' @references Schmeider, D. (1969). The Nucleolus of a Characteristic Function Game. \emph{SIAM Journal on Applied Mathematics}, 17(6), 1163–1170.
#' @references Software by J. Derks (Copyright 2005 Universiteit Maastricht, dept. of Mathematics), available in package \emph{MatTuGames},
#'
#'             \url{https://www.shorturl.at/i6aTF}.
#' @seealso \link{excesses}, \link{kohlbergcriterion}, \link{leastcore}, \link{nucleoluspcvalue}, \link{nucleolusvalue}
#' @export

prenucleolusvalue <- function(v, binary = FALSE, tol = 100*.Machine$double.eps) {

  # PRENUCLEOLO computes the prenucleolus of a coalitional game
  # INPUT
  # v = coalitional game (binary)
  # tol = Tolerance level (default, 100eps)
  #
  # OUTPUT
  #the prenucleolus of game v
  #########################################
  nC <- length(v)
  n <- log2(nC + 1)

  if(n>floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }

  if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, lo pasamos a binario.
    v <- lex2bin(v)
  }

  # Inicializamos el procedimiento
  # Asignación inicial,  x=solucion igualitaria
  x <- rep(v[nC] / n, n)
  # Coalicion que se satura
  Fam <- nC  # Coalición {1,...,n}, nC=[1 ... 1]=x1+...+xn

  # Otros valores iniciales
  FIN <- 0 #FIN = 1 ------ Nucleolus found

  while (FIN == 0) {

    #As long as FIN=0, another iteration
    ####################################
      # Queremos resolver el problema de optimización:
      # min{t : Fx=v(F), Ax+ut\geq v(S) }
    #
    ####################################
    # Las otras coaliciones: f==A

    f <- AfromFam(Fam, n) # Independent coalitions from those in family Fam

    # Resolvemos el sistema zFam+uA(j0)=0, u1=1.
    # Primer paso
    # Sea t0=max{v(S)-x(S): S=1,...,nC-1}=-min{x(S)-v(S):S=1,...,nC-1}
    # y sea j0 el menor índice donde se alcanza t0. Luego j0 determina una
    # coalición A0.
    # Estos valores pueden calcularse con pasoptimo(x,y,t,v,f)
    # para x,y=0,t=0,v,f


    # Paso inicial
    y <- rep(0, n)
    t <- 0
    paso <- pasoptimo(x, y, t, v, f, tol)
    t1 <- paso$lambda
    A <- paso$jj
    # Actualizamos el exceso, que se alcanza en la coalición A
    t <- -t1
    control <- 0 # control = 1 leastcore allocation found

    while (control == 0) {
      #  Utilizamos la función [y,flag,sat]=sistemaFamA(Fam,j0,n)
      sistema <- sistemaFamA(Fam, A, n, tol)
      y <- sistema$y
      exit <- sistema$exit
      sat <- sistema$sat

      if (exit == -1) { # Si alcanzamos el optimo
        control <- 1
        # sat ya está calculado
      } else if (exit == 0) { # Si no alcanzamos el optimo porque exit=0
        # Calculamos el nuevo paso
        paso <- pasoptimo(x, y, t, v, f, tol)
        lambda <- paso$lambda
        hatj <- paso$jj
        # Nueva solucion factible
        x <- x + lambda * y
        t <- t - lambda
        # Nueva matriz A
        A <- c(A, hatj)
      } else { # Si no alcanzamos el optimo y exit>=1
        # Calculamos el nuevo paso
        paso <- pasoptimo(x, y, t, v, f, tol)
        lambda <- paso$lambda
        hatj <- paso$jj
        #  Nueva solucion factible
        x <- x + lambda * y
        t <- t - lambda
        # Nueva matriz A, cambiamos la fila exit por la nueva donde se
        # alcanza el nuevo paso
        A[exit] <- hatj
      }
    } # fin del bucle while
    # LEASTCORE allocation x found along with vector sat
    Fam <- CoalitionsInd(Fam, sat, n) # Añadimos las coaliciones saturadas a Fam
    # y nos quedamos con las linealmente independientes, es decir,
    # las coaliciones independientes en la union de Fam y sat
    if (length(Fam) == n) {
      FIN <- 1
    }
  } # fin del while

  return(x)
}
