#' @title Perfect core game
#' @description This function returns the perfect core game with a given number of players.
#' @param n A number of players, as an integer.
#' @param binary A logical value. By default, \code{binary=FALSE}.
#' @return The characteristic function of the perfect core game with \code{n} players, as a vector in binary order if \code{binary=TRUE} and in lexicographic order otherwise.
#' @details The perfect core game of \eqn{n} players is defined by \deqn{v_P(S)=s-\sqrt{\frac{s(n-s)}{n-1}}\text{ for all }S\in 2^N,} where \eqn{s=|S|}.
#' @examples
#' perfectcoregame(6)
#' @references Shapley, L. S. (1971). Cores of convex games. \emph{International Journal of Game Theory}, 1(3), 11-26.
#' @export

perfectcoregame = function(n,binary=FALSE) {# Principio de la función

  ################################
  ###Comprobación datos entrada###
  ################################

  if (n!=floor(n))
  {
    stop("'n' must be an integer.")
  }

  ################################
  #####Creación de variables######
  ################################

  v=c()
  nC=2^n-1 #Número de coaliciones

  ################################
  ######Cuerpo de la función######
  ################################

  for(S in 1:((nC-1)/2))#Por dualidad, recorremos solo la mitad de las coaliciones.
  {
    num=S #Primero calculo los jugadores pertenecientes a la coalición S.
    J=floor(log2(num))+1
    Sjug=rep(0,n) # Inicializo la coalición S

    for (ii in 1:J)
    {
      if (num %% 2 ==1)
      {
        Sjug[ii]=1
      }
      num=num %/% 2
    }
    s=sum(Sjug)
    v[S]=s-sqrt(s*(n-s)/(n-1)) #vp(S)=s-sqrt(s(n-s)/(n-1))
    #Aplicamos dualidad
    nmenoss=n-s
    v[nC-S]=v[S]+n-2*s
  }
  v[nC]=n

  ################################
  ######Salidas de la función#####
  ################################
  if (binary == FALSE) { # Devolvemos el juego en orden lexicográfico.
    return(bin2lex(v))
  } else { # Devolvemos el juego en orden binario.
    return(v)
  }
}# Fin de la función
