#' @title Owen value
#' @description Given a game and a partition of the set of players, this function computes the Owen value.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @param partition A partition of the set of players, as a list of vectors. When not specified, it is taken to be the partition whose only element is the set of all players.
#' @param game A logical value. By default, \code{game=FALSE}. If set to \code{TRUE}, the associated quotient game is also returned.
#' @return The corresponding Owen value, as a vector; and, if \code{game=TRUE}, the associated quotient game, as a vector in binary order if \code{binary=TRUE} and in lexicographic order otherwise.
#' @details Let \eqn{v \in G^{N}} and let \eqn{C=\{C_{1},\dots,C_{m}\}} be a partition of the set of players.
#' For each \eqn{T \in 2^{N} \setminus \emptyset}, let \eqn{R'_{T}=\{j : C_{j} \cap T \neq \emptyset\}} and \eqn{R^{T}_{j}=C_{j} \cap T} for each \eqn{j \in \{1,\dots,m\}}.
#' Being \eqn{c_{T}} the Harsanyi dividend of coalition \eqn{T \in 2^{N}}, the Owen value of each player \eqn{i \in N} is defined as
#' \deqn{O_{i}(v,C)=\sum_{T \in 2^{N}:j \in R'_{T},i \in R^{T}_{j}}\frac{c_{T}}{|R'_{T}||R^{T}_{j}|}.}
#' @examples
#' v <- c(0,0,0,0,30,30,40,40,50,50,60,70,80,90,100) # in lexicographic order
#' owenvalue(v, partition=list(c(1,3),c(2),c(4)))
#' owenvalue(v)
#' round(owenvalue(v),10) == round(shapleyvalue(v),10)
#' w <- c(0,0,0,0,0,10,10,20,10,20,10,20,10,20,10,20,40,20,40,20,40,
#'        20,40,20,20,80,60,80,80,60,100) # in lexicographic order
#' owenvalue(w, partition=list(c(1,2,3),c(4,5)))
#' @references Owen, G. (1977). Values of Games with a Priori Unions. In R. Henn and O. Moeschlin (Eds.), \emph{Mathematical Economics and Game Theory} (pp. 76-88), Springer.
#' @seealso \link{shapleyvalue}, \link{harsanyidividend}
#' @export

owenvalue <- function(v, binary=FALSE, partition=NULL, game=FALSE) {

  # comprobaciones de siempre
  nC=length(v) # Número de coaliciones
  n=log(nC+1)/log(2) # Número de jugadores
  if (n>floor(n))
  {
    stop("'v' must have length 2^n-1 for some n.")
  }
  # fin comprobaciones de siempre

  # comprobaciones partition
  if (is.null(partition)) { # si no se proporciona partition, consideramos N
    partition <- list(getcoalition(2^n-1))
    m <- length(partition) # número de elementos en la partición
    # Convertimos los elementos de la partición en números binarios.
    P=rep(0,m) # inicializamos P
    for (i in 1:m){
      P[i]=getcoalitionnumber(partition[[i]])
    }
  } else { # si se proporciona partition, comprobamos que realmente lo es
    m <- length(partition) # número de elementos en la partición
    # Convertimos los elementos de la partición en números binarios.
    P=rep(0,m) # inicializamos P
    for (i in 1:m){
      P[i]=getcoalitionnumber(partition[[i]])
    }
    # Control: nos han dado una partición?
    if (sum(P) + 1 != 2^n) {
      stop("'partition' is not a partition of the player set N.")
    }
  }
  # fin comprobaciones partition

  #################################
  ### cálculo del valor de Owen ###
  #################################

  Ow <- numeric(n)

  if (binary == FALSE) { # Cuentas en lexicográfico.
    for (S in 1:(2^n - 1)) { # recorremos todas las coaliciones
      # Calculamos el HD de S.
      HD <- harsanyidividend(v=v,S=codebin2lex(n=n,Nbin=S),binary=binary)
      # Calculamos la intersección de S con cada elemento de la partición.
      if (HD != 0) {
        ISP <- bitwAnd(S, P)
        ISP <- ISP[ISP != 0] # elementos de la partición que cortan a S
        nSP <- length(ISP) # número de elementos que cortan a S
        # Calculamos aportación del dividendo de Harsanyi para cada jugador de ISP
        for (ii in seq_len(nSP)) { # para cada elemento de ISP
          jugISP <- as.logical(bitwAnd(2^(0:(n-1)), ISP[ii]))
          nISP <- sum(jugISP) # cardinal
          Ow <- Ow + (HD / (nSP * nISP)) * jugISP
        }
      }
    }
  } else { # Cuentas originales en binario.
    for (S in 1:(2^n - 1)) { # recorremos todas las coaliciones
      # Calculamos el HD de S.
      HD <- harsanyidividend(v=v,S=S,binary=binary)
      # Calculamos la intersección de S con cada elemento de la partición.
      if (HD != 0) {
        ISP <- bitwAnd(S, P)
        ISP <- ISP[ISP != 0] # elementos de la partición que cortan a S
        nSP <- length(ISP) # número de elementos que cortan a S
        # Calculamos aportación del dividendo de Harsanyi para cada jugador de ISP
        for (ii in seq_len(nSP)) { # para cada elemento de ISP
          jugISP <- as.logical(bitwAnd(2^(0:(n-1)), ISP[ii]))
          nISP <- sum(jugISP) # cardinal
          Ow <- Ow + (HD / (nSP * nISP)) * jugISP
        }
      }
    }
  }

  #############################
  ### salidas de la función ###
  #############################

  if (game == FALSE) { # Si el usuario no pide el juego cociente, directamente no lo calculamos.
    return(Ow)
  } else { # Si el usuario pide el juego cociente, lo calculamos y lo sacamos.

    # CÁLCULO JUEGO COCIENTE (VERSIÓN ESTELA)
    m <- length(partition)   # número de elementos en la partición
    # Número binario de los elementos de la partición
    P <- numeric(m) # inicializamos P
    for (p in 1:m) {
      P[p] <- sum(2^(unlist(partition[[p]]) - 1))
    }
    vP <- numeric(2^m - 1)
    for (S in 1:(2^m - 1)) { # Recorro todas las coaliciones
      num <- S
      # Obtengo los jugadores de S (como cadena de 0's y 1's)
      # Mayor jugador en S
      s <- floor(log2(num)) + 1
      # Inicializo. S tiene como máximo m jugadores
      #JugS <- as.numeric(intToBits(S)[1:s])
      JugS <- numeric(m)
      for (ii in 1:s) {
        if (num %% 2 == 1) {
          JugS[ii] <- 1
        }
        num <- (num - num %% 2) / 2
      }
      # Idea de Estela.
      # P partición: el número de la unión es la suma de los números
      vP[S] <- v[sum(P * JugS)]
    }
    # FIN CÁLCULO DEL JUEGO COCIENTE

    if (binary == FALSE) { # Devolvemos el juego cociente en orden lexicográfico.
      return(list(value=Ow,game=bin2lex(vP)))
    } else { # Devolvemos el juego cociente en orden binario.
      return(list(value=Ow,game=vP))
    }
  }

}
