#' @title Museum pass game
#' @description This function returns the characteristic function of the described museum pass game.
#' @param V A matrix of zeros and ones where each row represents a museum and each column represents a visitor. If museum \eqn{i} is visited by visitor \eqn{j}, \eqn{V_{ij}=1}; otherwise, \eqn{V_{ij}=0}.
#' @param p A vector containing the price that each visitor pays for their pass. By default, it is a vector of ones.
#' @param binary A logical value. By default, \code{binary=FALSE}.
#' @return The characteristic function of the museum pass game, as a vector in binary order if \code{binary=TRUE} and in lexicographic order otherwise.
#' @details Let \eqn{N} be a non-empty and finite set of museums and let \eqn{U} be a non-empty and finite set of visitors.
#' The museum matrix, \eqn{V\in \{0,1\}^{N \times U}}, specifies which museums are visited by which visitors: \eqn{V_{ij}=1} if and only if museum \eqn{i\in N}
#' is visited by visitor \eqn{j\in U}. The vector \eqn{p\in\mathbb{R}^|U|_+} represents, for each visitor \eqn{j}, the price they pay for their museum pass
#' (all passes are equal, in the sense that they grant access to the same set of museums, but the price may not be the same for all visitors).
#'
#' The total revenue is to be divided among the museums. Given a museum pass situation \eqn{(N,U,V,p)}, the museum pass game is defined by
#' \deqn{v(S)=\sum_{j\in U:N^j\subset S}p_j \ \text{for each coalition } S \in 2^{N},}
#' where \eqn{N^j=\{i\in N:V_{ij}=1\}} is the set of museums visited by \eqn{j\in U}.
#' @examples
#' V <- rbind(c(1,0,1,1,0), c(0,1,1,1,0), c(1,1,0,0,1), c(1,0,1,0,1))
#' museumpassgame(V, p=c(1,1,4,5,8))
#' @references Ginsburgh, V. & Zang, I. (2003). The museum pass game and its value. \emph{Games and economic behavior}, 43(2), 322-325.
#' @export
museumpassgame=function(V, p=rep(1,dim(V)[2]), binary=FALSE){ # Principio de la función

  ################################
  ### Comprobación datos entrada##
  ################################

  if(sum(V>1)>0 | sum(V<0)>0){
    stop("'V' must be a matrix of ones and zeros.")
  }

  if(sum(p<=0)>0){
    stop("Every element in 'p' must be positive.")
  }

  if(length(p)<dim(V)[2] | length(p)>dim(V)[2]) {
    stop("The length of 'p' must match the number of columns in 'V'.")
  }

  ################################
  #####Creación de variables######
  ################################

  n=dim(V)[1] #Número de jugadoores (museos)
  nC=2^n-1 #Número de coaliciones.
  v=rep(0,nC) #Inicializamos el juego.

  ################################
  ######Cuerpo de la función######
  ################################

  for(S in 1:(nC))
  {
    #Calculamos los jugadores pertenecientes a la coalición:
    num=S
    J=floor(log2(num))+1
    Sjug=c() # Inicializo la coalición S

    for (ii in 1:J)
    {
      if (num %% 2 ==1)
      {
        Sjug=c(Sjug,ii)
      }
      num=num %/% 2
    }
    for(j in 1:dim(V)[2])
    {#Vamos a ver si esta coalición tiene clientes únicos.
      if(sum(V[Sjug,j])>=1 & sum(V[-Sjug,j])==0)
      {#Si tiene clientes únicos, asignamos su pago a la coalición.
        v[S]=v[S]+p[j]
      }
    }
  }

  ################################
  ######Salidas de la función#####
  ################################
  if (binary == FALSE) { # Devolvemos el juego en orden lexicográfico.
    return(bin2lex(v))
  } else { # Devolvemos el juego en orden binario.
    return(v)
  }
}# Fin de la función
