#' @title Monotonic check
#' @description This function checks if the given game is monotonic.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @param instance A logical value. By default, \code{instance=FALSE}.
#' @return \code{TRUE} if the game is monotonic, \code{FALSE} otherwise. If \code{instance=TRUE} and the game is not monotonic, the function also returns the positions (binary order positions if \code{binary=TRUE}; lexicographic order positions otherwise) of a pair of coalitions violating monotonicity.
#' @details A game \eqn{v\in G^N} is monotonic if \eqn{v(S) \le v(T)} for all \eqn{S, T \in 2^N} such that \eqn{S\subset T}.
#' @examples
#' v <- c(0, 0, 1, 5, 1, 1, 2)
#' monotoniccheck(v, binary=FALSE, instance=TRUE)
#' @seealso \link{additivecheck}, \link{superadditivecheck}, \link{zeromonotoniccheck}
#' @export
monotoniccheck <- function(v, binary = FALSE, instance = FALSE) { # Principio de la función

  ################################
  ### Comprobación datos entrada###
  ################################

  nC <- length(v) # Número de coaliciones.
  n <- log(nC + 1) / log(2) # Número de jugadores.
  if (n > floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }
  if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, lo pasamos a binario.
    v <- lex2bin(v)
  }

  ################################
  ##### Creación de variables######
  ################################

  checkR <- TRUE # Por defecto, mientras no encuentre un contraejemplo, checkR=TRUE
  exampleR <- NULL # Por defecto, no hay contraejemplo
  S <- 1 # Inicializo coalición S.
  Jug <- c(1:n) # Vector de jugadores
  tol <- 100*.Machine$double.eps

  # S para el número de coalición. En este caso Sjug para los jugadores.
  ################################
  ###### Cuerpo de la función######
  ################################

  while (checkR & S <= (nC - 1)) { # Mientras no haya encontrado un contraejemplo y no haya superado el número máximo de coaliciones, hago la comprobación.
    num <- S # Primero calculo el vector con los jugadores pertenecientes a la coalición S.
    J <- floor(log2(num)) + 1
    Sjug <- c() # Jugadores pertenecientes a S

    for (ii in 1:J)
    {
      if (num %% 2 == 1) {
        Sjug <- c(Sjug, ii)
      }
      num <- num %/% 2
    }

    for (jj in 1:length(Jug[-Sjug]))
    { # Estudio solo las coaliciones a las que se le añade un elemento. Las siguientes se realizan en iteraciones posteriores.
      Rjug <- c(Sjug, Jug[-Sjug][jj]) # Jugadores en la coalición R.
      R <- sum(2.^(Rjug - 1)) # Número que ocupa la coalición R en el juego.
      if (v[R] - v[S] + tol < 0) # v(S)>v(R)
        {
          checkR <- FALSE
          exampleR <- c(S, R)
          break
        }
    }
    S <- S + 1
  }

  ################################
  ###### Salidas de la función#####
  ################################

  if (instance == FALSE) { # Si el usuario sólo pide una salida, es decir, instance=FALSE
    return(check = checkR)
  } else { # Si instance=TRUE, la función devuelve también el código numérico de las coaliciones S y R.
    if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, devolvemos las posiciones lexicográficas de un par de coaliciones que falla.
      return(list(check = checkR, example = c(codebin2lex(n, exampleR[1]), codebin2lex(n, exampleR[2]))))
    } else { # Si el juego se introdujo en binario, devolvemos las posiciones binarias de un par de coaliciones que falla.
      return(list(check = checkR, example = exampleR))
    }
  }
} # Fin de la función
