#' @title Dual game
#' @description Given the characteristic function of a game, this function returns the characteristic function of the dual game.
#' @param v A characteristic function, as a vector.
#' @return The characteristic function of the dual game. It is to be interpreted according to the order that \code{v} is introduced in.
#' @details The dual game of \eqn{v\in G^N} is defined by \eqn{v^D(S)=v(N)-v(N\backslash S)} for all \eqn{S \in 2^N}.
#' @examples
#' v <- c(rep(0,4),rep(5,6),rep(20,4),40)
#' dualgame(v)
#' v <- seq(1:31)
#' dualgame(v)
#' dualgame(dualgame(v)) == v
#' @export

dualgame <- function(v) {

  # comprobación datos entrada
  nC <- length(v) # número de coaliciones
  n <- log(nC + 1) / log(2) # número de jugadores
  if (n > floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }

  # definimos y calculamos el dual
  dual <- v
  for (S in 1:((nC - 1) / 2)) # por dualidad, recorremos solo la mitad de las coaliciones
  {
    dual[S] <- v[nC] - v[nC - S] # v^c(S)=v(N)-v(N\S)
    dual[nC - S] <- v[nC] - v[S] # v^c(N\S)=v(N)-v(S)
  }

  # salida de la función
  return(dual)

} # Fin de la función
