#' @title Balanced check
#' @description This function checks if the given game is balanced and computes its balanced cover.
#' @param v A characteristic function, as a vector.
#' @param game A logical value. By default, \code{game=FALSE}. If set to \code{TRUE}, the balanced cover of the game is also returned.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @param tol A tolerance parameter, as a non-negative number.\cr
#'            By default, \code{tol=100*.Machine$double.eps}.
#' @return \code{TRUE} if the game is balanced, \code{FALSE} otherwise. If \code{game=TRUE}, the balanced cover of the game is also returned.
#' @details Let \eqn{v \in G^{N}}. A family \eqn{F} of non-empty coalitions of \eqn{N}
#' is balanced if there exists a weight family \eqn{\delta^{F} = \{ \delta^{F}_{S} \}_{S \in F}} such that
#' \eqn{\delta^{F}_{S} > 0} for each \eqn{S \in F} and \eqn{\sum_{S \in F} \delta^{F}_{S} e^{S} = e^{N}},
#' being \eqn{e^{S}} the characteristic vector of \eqn{S}, that is, the vector \eqn{(e_{i}^{S})_{i \in N}}
#' in which \eqn{e_{i}^{S}=1} if \eqn{i \in S} and \eqn{e_{i}^{S}=0} if \eqn{i \notin S}).
#'
#' The game \eqn{v} is balanced if, for each balanced family \eqn{F}, it is true that
#' \deqn{\sum_{S \in F} \delta^{F}_{S} v(S) \leq v(N).}
#'
#' The balanced cover of \eqn{v} is the game \eqn{\tilde{v}} defined by
#' \eqn{\tilde{v}(S)=v(S)} for all \eqn{S \neq N} and
#' \deqn{\tilde{v}(N) = \max_{\delta \in P}{\sum_{S \subset N} \delta_{S} v(S)},}
#' being \eqn{P} the set of the weight families associated with the balanced families of \eqn{N}.
#'
#' A game is balanced if and only if it coincides with its balanced cover.
#' By the Bondareva-Shapley Theorem, a game has a non-empty core if and only if it is balanced.
#' @examples
#' balancedcheck(c(12,10,20,20,50,70,70), game=TRUE)
#' balancedcheck(c(rep(0,4), rep(30,6), rep(0,4), 50))
#' v <- runif(2^3-1,0,10) # random three-player game
#' balancedcheck(v, game=TRUE)
#' balancedcheck(balancedcheck(v, game=TRUE)$game) # balanced cover is indeed balanced
#' balancedcheck(runif(2^(15)-1,min=10,max=20)) # random game
#' @references Maschler, M., Solan, E., & Zamir, S. (2013). \emph{Game Theory}. Cambridge University Press.
#' @seealso \link{totallybalancedcheck}
#' @export

balancedcheck <- function(v, game = FALSE, binary = FALSE, tol = 100*.Machine$double.eps) {
  # BALANCEDCHECK Comprueba si un juego v es o no equilibrado.
  # v: función característica del juego, en orden binario.
  # Devuelve:
  # - Bv: 1 si v es equilibrado, 0 si no lo es
  # - bcover: envoltura equilibrada de v

  #  Resolvemos:
  # max \sum \delta_S v(S) , S no vacío
  # s.a. \sum _{S: i\in S} \delta_S =1, para todo i\in N.

  # Programamos a definición do balanced cover de v que está no libro
  # de Mashler (o dual): dado un xogo non equilibrado v, calculamos o menor v(N)
  # que fai que o núcleo sexa non baleiro.
  # output: si v es equilibrado y la envoltura equilibrada
  # Ejemplo: v <- c(0,0,0,3,15,6,10) # no equilibrado (está en lex). La envoltura equilibrada es 0  0    0 3 15  6 15
  # v <- c(rep(0,4),rep(30,6),rep(0,4),50), en lex. La envoltura equilibrada es 0  0  0  0 30 30 30 30 30 30  0  0  0  0 60
  # v <- c(12,10,20,20,50,70,70), $bcover  12 10 20 20 50 70 82. Su core es un segmento
  # v <- c(30,40,70,10,20,5,70) $bcover 30  40  10  70  20   5 140. Su core es un punto

  nC <- length(v)
  n <- log2(nC + 1)
  if (n > floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }
  v.copia <- v # guardamos v tal como nos lo dieron, nos vendrá bien al final
  if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, lo pasamos a binario.
    v <- lex2bin(v)
  }

  Indis <- 2^(0:(n - 1)) # Posiciones de las coaliciones individuales en binario
  vIndis <- v[Indis] # Valores de las coaliciones individuales
  coaliNonIndi <- c(1:nC)
  coaliNonIndi <- coaliNonIndi[-Indis] # Posiciones del resto de las coaliciones

  # 0-normalización y matriz inicial del sistema (sin función objetivo)
  v0 <- rep(0, nC) # Inicialización de la 0-normalización
  A <- matrix(0, nrow = nC, ncol = n) # Inicialización de la matriz inicial

  for (S in coaliNonIndi) { # En A só precisamos as coalicións non individuais
    bits <- as.integer(intToBits(S))[1:n]
    A[S, ] <- bits
    v0[S] <- v[S] - sum(vIndis * bits) # 0 normalizamos
  }

  ###################
  # Matriz completa, con función objetivo
  ###################
  A <- rbind(t(A), as.vector(v0))
  A <- A[, -Indis] # Eliminamos los costes individuales v0(i)
  Ab <- cbind(A, c(rep(1, n), 0))

  # Comezamos proceso (eliminación de columnas e comprobacións)
  if (ncol(Ab) >= 2) {
  pivfil <- max(Ab[n + 1, 1:(ncol(Ab) - 1)]) # buscamos la columna inicial con mayor coste
  entrac <- which.max(Ab[n + 1, 1:(ncol(Ab) - 1)])
  } else {
    pivfil <- 0  # salimos del bucle
  }
  pivotes <- 1

  while (pivfil > 0) { # O proceso repítese mentres algún dos custos (elementos da última fila) sexa positivo
    Z <- which(Ab[1:n, entrac] < tol)
    division <- Ab[1:n, ncol(Ab)] / Ab[1:n, entrac]
    division[abs(division) < tol] <- 0
    division[Z] <- Inf
    saef <- which.min(division) # Fila con menor ratio
    Aux <- Ab[saef, ] #  Aux é a fila na que está o pivote. Se conserva para las cuentas
    pivotes <- pivotes * Aux[entrac]

    for (ii in 1:(n + 1)) { # Poñemos un cero no resto de elementos da columna onde está o pivote
      Ab[ii, ] <- Aux[entrac] * Ab[ii, ] - Ab[ii, entrac] * Aux
    }

    Ab[saef, ] <- Aux
    Ab <- Ab[, -entrac, drop = FALSE]
    Ab[abs(Ab) < tol] <- 0 # Recupero a fila e elimino a columna na que estaba o pivote

    if (is.null(dim(Ab)) || ncol(Ab) < 2) {
      break  # Salir del bucle si no hay suficientes columnas
    } else {
      pivfil <- max(Ab[n + 1, 1:(ncol(Ab) - 1)])
      entrac <- which.max(Ab[n + 1, 1:(ncol(Ab) - 1)])
    }

  }

  vN <- Ab[n + 1, ncol(Ab)]
  axustefinal <- vN / pivotes # Divido o último elemento da táboa entre o produto dos pivotes

  Bv <- TRUE
  bcover <- v.copia # cubrimos a la vez los casos de que nos dieran v en bin y en lex

  if (abs(abs(axustefinal) - v0[nC]) > tol) {
    Bv <- FALSE
    bcover[nC] <- abs(axustefinal) + sum(v[Indis])
  }

  # Salidas.
  if (game == FALSE) {
    return(check = Bv)
  }
  return(list(check = Bv, game = bcover))

}
