
#include <photosynthesis_functions.hpp>

//' calc_photo
//'
//' Generates estimates of photosynthetic rates using a Farquar type photosynthesis model.
//'
//' This function is called internally by \code{get_input()}.
//'
//' @param leaf_temp  matrix with ntimesteps rows, nsites columns, containing leaf temperature [°C]
//' @param photo_active_rad  matrix with ntimesteps rows, nsites columns, containing photosynthetically active radiation [umol*m^(-2)*s^(-1)]
//' @param atmospheric_co2 matrix with ntimesteps rows, nsites columns, containing atmospheric CO2 partial pressure [Pa]
//' @param ppl a list of photosynthetic parameters (see \code{p3} and \code{p4})
//' @param pht a string indicating "c3" or "c4" photosynthesis
//' @return matrix with ntimesteps rows, nsites columns, containing the calculated photosynthetic rates [umol*m^(-2)*s^(-1)]
//' @export
// [[Rcpp::export]]
arma::mat calc_photo(const arma::mat& leaf_temp, const arma::mat& photo_active_rad, const arma::mat& atmospheric_co2, const Rcpp::List& ppl, Rcpp::StringVector pht){
  size_t nsites = leaf_temp.n_rows;
  size_t nt = leaf_temp.n_cols;
  size_t len = leaf_temp.size();

  if(photo_active_rad.n_rows != nsites || atmospheric_co2.n_rows != nsites ||
     photo_active_rad.n_cols != nt || atmospheric_co2.n_cols != nt){
    Rcpp::stop("ERROR: In Photosynthetic parameter calculation: Matrices for leaf temp, photosyntheticaly active radiation and atmospheric carbon do not have the same dimensions.");
  }

  arma::mat out(nsites, nt, arma::fill::none);
  std::string photo_type = Rcpp::as<std::string>(pht);
  photo_type[0] = std::tolower(photo_type[0]);
  if(photo_type == "c3"){
    ParameterVector<p3> pp3(ppl);
    ttr_photo::photo_state p3s(pp3);
    for(size_t i = 0; i < len; i++){
      out(i) = p3s(leaf_temp(i), photo_active_rad(i), atmospheric_co2(i));
    }
  }
  else if(photo_type == "c4"){
    ParameterVector<p4> pp4(ppl);
    ttr_photo::photo_state p4s(pp4);
    for(size_t i = 0; i < len; i++){
      out(i) = p4s(leaf_temp(i), photo_active_rad(i), atmospheric_co2(i));
    }
  }
  else {
    Rcpp::stop("No photosynthesis type given");
  }
  return out;
}
