% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/seasonal.R
\name{convert_monthly_to_seasonal}
\alias{convert_monthly_to_seasonal}
\title{Convert a data table from monthly to seasonal format}
\usage{
convert_monthly_to_seasonal(
  dt,
  vars = NULL,
  by = NULL,
  FUN = sum,
  ...,
  seasons = c("MAM", "JJAS", "OND"),
  only_complete_seasons = TRUE
)
}
\arguments{
\item{dt}{A data table containing the values for conversion.}

\item{vars}{Character vector of names of the columns containing the values for conversion. Default is to try converting everything that is not contained in \code{by} and that is not recognized as tercile category (see \code{\link[=tc_cols]{tc_cols()}}) or
tercile forecast (\code{'below'}, \code{'normal'}, \code{'above'}).}

\item{by}{Character vector of column names to group by. Separate values are derived for each unique combination of values in \code{by}.
Defaults to all \code{\link[=dimvars]{dimvars()}} that are not \code{'month'}, which is usually what you want.}

\item{FUN}{function for aggregation.}

\item{...}{arguments passed to \code{FUN}, for example \code{na.rm}.}

\item{seasons}{Vector of character strings specifying seasons. See details. Defaults to \code{c('MAM', 'JJAS', 'OND')}, which are the seasons considered in the COFs for the Greater Horn of Africa.}

\item{only_complete_seasons}{Logical. If \code{TRUE}, only values are kept for which we have data for all months. For example, no \code{OND} values would be derived if the data for December is missing.}
}
\description{
Converts monthly to seasonal data. The function default values are set for precipitation. In particular, default behavior is to \emph{sum}
values over all months contained in a season. If you want to average instead (for temperature, for example), you can change the aggregation function \code{FUN}.
}
\details{
Note that it is impossible to derive seasonal tercile categories from monthly ones (and similar for seasonal tercile forecasts). For getting these, you should convert to seasonal
\emph{before} deriving the tercile categories or forecasts, e.g. by using \code{\link[=add_tercile_cat]{add_tercile_cat()}} or \code{\link[=tfc_from_efc]{tfc_from_efc()}}.

Seasons are provided as sequences of capitalized initial characters of the months they contain, e.g. \code{'MAM'} for March-April-May.
They can have any length from 1 to 12 months and are allowed to overlap and wrap over the end of the year
(for example, you can provide \code{seasons = c('OND', 'NDJ')} to derive values for October-December and November-January).
If a season includes months from 2 years, it gets assigned the year of its starting month. For example, \code{season = 'NDJ'} and \code{year = 2000} refers to values for the season November 2000 to January 2001.

Factor- or Character-valued columns cannot be aggregated to seasonal values. If \code{vars} contains columns that are factor- or character-valued, it checks whether they take a unique value for each grouping level
provided in \code{by}. If yes, they are kept, else they are discarded. A typical case where this is useful is when your data table contains country names (see \code{\link[=add_country]{add_country()}}).
The grouping levels usually include \code{'lon'}, \code{'lat'}, so there is only one country name per grouping level and the name is kept.
}
\examples{
\donttest{
# returns empty data table, because the example data does not contain data for a full season:
dt = convert_monthly_to_seasonal(chirps_monthly)

# remove terc_cat first to avoid the warning,
# and set season to the months we actually have data for:
dt2 = convert_monthly_to_seasonal(copy(chirps_monthly)[,terc_cat := NULL], seasons = c('ND'))
print(dt2)

# season OND, get monthly averages rather than sums, and force the function to derive values
# even though we do not have October-data:
dt3 = convert_monthly_to_seasonal(chirps_monthly,
                                  seasons = c('OND'),
                                  FUN = mean,
                                  only_complete_seasons = FALSE)
print(dt3)
}
}
