source(paste(getwd(), "/R/SIMle.Legen.v1.R", sep = ""))
source(paste(getwd(), "/R/SIMle.Cheby.v1.R", sep = ""))
source(paste(getwd(), "/R/SIMle.Four.v1.R", sep = ""))
source(paste(getwd(), "/R/SIMle.Csp.v1.R", sep = ""))
source(paste(getwd(), "/R/SIMle.db1-20.v1.R", sep = ""))
source(paste(getwd(), "/R/SIMle.plot.v1.R", sep = ""))
source(paste(getwd(), "/R/SIMle.original_code.v1.R", sep = ""))




#' Automated exact form test
#' @description This function utilizes L2 test for the automated execution of exact form tests 
#'              with chosen bases.
#'
#' @param ts ts is the data set which is a time series data typically
#' @param c the maximum value of number of basis for time input
#' @param d the maximum value of number of basis for variate input
#' @param b_time type of basis for time input 
#' @param b_timese type of basis for variate input
#' @param mp_type select type of mapping function, "algeb" indicates algebraic mapping on the real line. "logari" represents logarithmic mapping on the real line
#' @param ops Criteria for choosing the number of bases are provided by the package, offering four options: "AIC," "BIC," "CV," and "Kfold," each corresponding to a specific Criteria
#'        "AIC" stands for Akaike Information Criterion, "BIC" stands for Bayesian Information Criterion, "CV" represents cross-validation, and "Kfold" corresponds to k-fold cross-validation 
#'        for time series data
#' @param exact_func A list contains elements that are matrix contain exact functions, which are desired to be tested. The k-th element represents the k-th variable. 
#'        The matrix contains values of the exact function within its domain
#' @param m the window size for the simultaneous confidence region procedure, with the default being 'MV,' which stands for the Minimum Volatility method 
#' @param r indicates number of variate 
#' @param s s is a positive scaling factor, the default is 1
#' @param per the percentage for test set used in cross validation option "CV"
#' @param k the number of fold used in k-fold cross validation "Kfold"
#' @param upper upper The upper bound for the variate basis domain. The default value is 10. When "algeb" or "logari" is chosen, the domain is automatically set from -upper to upper
#'
#' 
#' @details
#' In the parameter type, this package provides 32 types of bases, including options such as 'Legen' for Legendre polynomials, 'Cheby' for the first kind of Chebyshev polynomials, 
#' 'tri' for trigonometric polynomials, 'cos' for cosine polynomials, 'sin' for sine polynomials, and 'Cspli' for the class of spline functions. In the 'Cspli' option, the first input 
#' 'c' represents knots plus 2, which correspond to 0 and 1. The term 'or' indicates the order of splines, so the number of basis elements is the number of knots + 2 - 2 plus the number 
#' of the order. When functions automatically choose the number of basis elements for splines, the number is not less than the order of the spline. The package provides 'db1' to 'db20' 
#' for Daubechies1 wavelet basis to Daubechies20 wavelet basis, and 'cf1' to 'cf5' for Coiflet1 wavelet basis to Coiflet5 wavelet basis. The wavelet tables provided by the Sie2nts package 
#' are generated by the Cascade algorithm using a low-pass filter. If exact values of wavelets are required, the Recursion algorithm should be used.
#' 
#' @return A list whose elements are p value of exact form test. Each element in the list represents p-values in the order of variates.
#' @references \[1] Ding, Xiucai, and Zhou, Zhou. “Estimation and inference for precision matrices of nonstationary time series.” The Annals of Statistics 48(4) (2020): 2455-2477. 
#' @references \[2] Ding, Xiucai, and Zhou, Zhou. “Auto-regressive approximations to non-stationary time series, with inference and applications.” Available online, 2021.
#' @references \[3] Ding, Xiucai, and Zhou Zhou. "Simultaneous Sieve Inference for Time-Inhomogeneous Nonlinear Time Series Regression." Available online, 2021.
#' @importFrom Matrix kronecker
#' @importFrom Matrix bdiag
#' @import ggplot2
#' @importFrom plotly plot_ly
#' @importFrom plotly add_surface
#' @importFrom plotly layout
#' @importFrom plotly add_markers
#' @importFrom stats rnorm
#' @importFrom stats integrate
#' @import stringr
#' @import RCurl
#' @import splines
#' @import methods
#' @import utils
#' @import Sie2nts
#' @importFrom stats sd
#' @export





auto.exact.test <- function(ts, c, d, b_time, b_timese, mp_type, ops, exact_func, m = "MV", r = 1, s = 1, per = 0, k = 0, upper = 10){
  best_cd = best_cd.auto.fit(ts, c, d, b_time, b_timese, mp_type, ops, r = r, s = s, per = per, k = k)
  
  basis_candi = c("Legen","Cheby","tri", "cos", "sin", "Cspli", "db1", "db2", "db3", "db4", "db5",
                  "db6", "db7", "db8", "db9", "db10",
                  "db11", "db12", "db13", "db14", "db15",
                  "db16", "db17", "db18", "db19", "db20",
                  "cf1", "cf2", "cf3", "cf4", "cf5"
  )
  
  #if(m == "MV"){ # refer to minimum volatility (MV) method
  #  m =  16 # min_vola()
  #} 
  
  if((b_time %in% basis_candi) && (b_timese %in% basis_candi)){
    res = exact_form_test(ts, as.numeric(best_cd[1]), as.numeric(best_cd[2]), m, b_time, b_timese, mp_type, exact_func, r = r, s = s, upper = upper)
    return(res)
  } else{
    return(stop("Invalid option!"))
  }
}
