#' Get coordinates of neighbors of specified distance
#'
#' \code{getNeighborCoords} - Returns coordinates of all pixels in a two
#' dimensional raster/image/... with a specified Euclidean distance from the
#' center, i.e., a pixel circle, either a ring or the whole area with a
#' specified radius.
#'
#' @param center Numeric vector of length two specifying the coordinates
#' (in pixels) of the center point.
#' @param radius Integer value specifying the radius (in pixels). A radius of
#' 0 returns only the center point itself.
#' @param dims_px Numeric vector of length two (default: 1,000 x 1,000)
#' specifying the dimensions (in pixels). Neighbors outside are ignored.
#' @param type Character, "ring" (default) or "area", specifying if the
#' pixel coordinates of only the outer ring or of the whole circle area should
#' be returned.
#' @param tol Character, "strict" (default) or "loose", specifying how thick
#' the ring/outer edge of the area should be, i.e., if pixels
#' close but not exactly on the circle should be included. If set to "loose",
#' all pixels that touch the circle are included. If set to "strict", only
#' additional pixels whose centers have a distance of at most \code{tol_val}
#' to the circle are included.
#' @param tol_val Numeric value specifying the tolerance value (>=0,<1,
#' default: 0.5), i.e., pixels are considered neighbors if the difference
#' between their Euclidean distance to the center and the radius is less than
#' or equal to the tolerance value.
#' Only applies if \code{tol} is set to "strict". \cr
#' A radius of 1 with a tolerance of 0.5 returns all 8 surrounding pixels and
#' with a tolerance of 0 only the 4 orthogonally neighboring pixels.
#'
#' @return \code{getNeighborCoords} Numeric matrix with two columns containing
#' the neighbors' coordinates (in pixels).
#'
#' @export
#' @rdname getNeighborCoords
#'
#' @examples
#' # The neighbors with radius 1 of point (4,4) in an 8x8 grid.
#' # With tolerance 0.5:
#' test <- matrix(0, nrow = 8, ncol = 8)
#' test[getNeighborCoords(c(4,4), 1, c(8,8))] <- 1
#' test
#' # With tolerance 0.4:
#' test <- matrix(0, nrow = 8, ncol = 8)
#' test[getNeighborCoords(c(4,4), 1, c(8,8), tol_val = 0.4)] <- 1
#' test
getNeighborCoords <- function(center, radius, dims_px = c(1000,1000),
                               type = "ring", tol = "strict", tol_val = 0.5){
  if(sum(c(center, dims_px)<=0)>0){
    stop("'center' and 'dims_px' must be positive.")
  }
  if(radius<0){
    stop("'radius' must be non-negative.")
  }
  if(tol_val<0 || tol_val >=1){
    stop("'tol_val' must be >=0 and <1.")
  }
  radius <- round(radius)
  if(!type %in% c("ring", "area")){
    stop("Unknown value for 'type', Available are 'ring' and 'area'.")
  }
  if(!tol %in% c("strict", "loose")){
    stop("Unknown value for 'tol', Available are 'strict' and 'loose'.")
  }

  # Initialize an empty matrix to store valid neighbor coordinates
  neighbors <- matrix(ncol = 2, nrow = 0)

  for (nx in 0:(floor(radius/2)+1)) {
    if(nx==0){
      # For the center add all pixels above and below until reaching
      # the circle edge. -------------------------------------------------------
      y_circ <- radius
    } else {
      # For the other x-positions compute the maximal value in this column on
      # the half circle. -------------------------------------------------------
      if(tol == "strict"){
        y_middle <- round(sqrt(radius^2-nx^2))
        y_up <- round(sqrt(radius^2-(nx-0.5)^2))
        y_low <- round(sqrt(max(c(0,radius^2-(nx+0.5)^2))))
        # Only include pixel (nx,y_up) if its distance is smaller than the
        # radius + tolerance.
        if(y_up != y_middle && abs(sqrt((nx)^2 + (y_up)^2) - radius) > tol_val){
          # If the left value is y_up, then the pixel below is near the ring.
          y_up <- max(y_middle, y_up-1)
        }
        if(y_low != y_middle && abs(sqrt((nx)^2 + (y_low)^2) - radius) > tol_val){
          # If the right value is y_low, then the pixel above is near the ring.
          y_low <- min(y_middle, y_low+1)
        }
        y_circ <- y_up:y_low
      } else { # "loose"
        # Add the y-values for the left and right edges of the pixel.
        y_circ <- round(sqrt(radius^2-(nx-0.5)^2)):
                  round(sqrt(max(c(0,radius^2-(nx+0.5)^2))))
      }
    }
    y_circ <- unique(round(y_circ))
    if(type == "ring"){
      for(y_c in y_circ){
        # For the ring, only add edge coordinates...
        neighbors <- rbind(neighbors,
                            # row + y     ,  col + x
                           c(center[1]+y_c, center[2]-nx),
                           c(center[1]+y_c, center[2]+nx),
                           c(center[1]-y_c, center[2]-nx),
                           c(center[1]-y_c, center[2]+nx))
        # ... but also exchange x and y to get the whole ring.
        neighbors <- rbind(neighbors,
                           # row + x     ,  col + y
                           c(center[1]+nx, center[2]-y_c),
                           c(center[1]+nx, center[2]+y_c),
                           c(center[1]-nx, center[2]-y_c),
                           c(center[1]-nx, center[2]+y_c))
      }
    } else { # "area"
      # For the area add everything in-between...
      neighbors <- rbind(neighbors,
                                    # row + y
                           matrix(c(center[1]+(-max(y_circ)):max(y_circ),
                                    center[1]+(-max(y_circ)):max(y_circ),
                                    # col + x
                                    rep(center[2] + nx, 2*max(y_circ)+1),
                                    rep(center[2] - nx, 2*max(y_circ)+1)),
                                  ncol = 2, byrow = FALSE))
      # ... but also exchange x and y to get the sides of the area.
      neighbors <- rbind(neighbors,
                                  # row + x
                         matrix(c(rep(center[1] + nx, 2*max(y_circ)+1),
                                  rep(center[1] - nx, 2*max(y_circ)+1),
                                  # col + y
                                  center[2]+(-max(y_circ)):max(y_circ),
                                  center[2]+(-max(y_circ)):max(y_circ)),
                                ncol = 2, byrow = FALSE))
    }
  }

  # Remove coordinates outside and duplicates. Then, return the matrix.
  neighbors <- neighbors[(neighbors[,1]>0 & neighbors[,1]<=dims_px[1] &
                          neighbors[,2]>0 & neighbors[,2]<=dims_px[2]),]
  neighbors <- unique(neighbors)
  return(neighbors)
}
