% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ata.R, R/methods.R
\name{ata}
\alias{ata}
\alias{ata_relative_objective}
\alias{ata_absolute_objective}
\alias{ata_constraint}
\alias{ata_item_use}
\alias{ata_item_enemy}
\alias{ata_item_fix}
\alias{ata_solve}
\alias{print.ata}
\alias{plot.ata}
\title{Automated Test Assembly (ATA)}
\usage{
ata(pool, n_forms = 1, test_len = NULL, max_use = NULL, ...)

ata_relative_objective(x, coef, mode = c("max", "min"), tol = NULL,
  negative = FALSE, forms = NULL, collapse = FALSE,
  internal_index = FALSE)

ata_absolute_objective(x, coef, target, equal_tol = FALSE,
  tol_up = NULL, tol_down = NULL, forms = NULL, collapse = FALSE,
  internal_index = FALSE)

ata_constraint(x, coef, min = NA, max = NA, level = NULL,
  forms = NULL, collapse = FALSE, internal_index = FALSE)

ata_item_use(x, min = NA, max = NA, items = NULL)

ata_item_enemy(x, items)

ata_item_fix(x, items, min = NA, max = NA, forms)

ata_solve(x, solver = c("lpsolve", "glpk"), return_format = c("model",
  "form", "simple"), silent = FALSE, time_limit = 10,
  message = FALSE, ...)

\method{print}{ata}(x, ...)

\method{plot}{ata}(x, ...)
}
\arguments{
\item{pool}{the item pool(s), a list of '3pl', 'gpcm', and 'grm' items}

\item{n_forms}{the number of forms to be assembled}

\item{test_len}{test length of each form}

\item{max_use}{maximum use of each item}

\item{...}{options, e.g. group, common_items, overlap_items}

\item{x}{an ATA object}

\item{coef}{the coefficients of the objective function}

\item{mode}{optimization direction: 'max' for maximization and 'min' for minimization}

\item{tol}{the tolerance paraemter}

\item{negative}{\code{TRUE} when the objective function is expected to be negative}

\item{forms}{forms where objectives are added. \code{NULL} for all forms}

\item{collapse}{\code{TRUE} to collapse into one objective function}

\item{internal_index}{\code{TRUE} to use internal form indices}

\item{target}{the target values of the objective function}

\item{equal_tol}{\code{TRUE} to force upward and downward tolerance to be equal}

\item{tol_up}{the range of upward tolerance}

\item{tol_down}{the range of downward tolerance}

\item{min}{the lower bound of the constraint}

\item{max}{the upper bound of the constraint}

\item{level}{the level of a categorical variable to be constrained}

\item{items}{a vector of item indices, \code{NULL} for all items}

\item{solver}{use 'lpsolve' for lp_solve 5.5 or 'glpk' for GLPK}

\item{return_format}{the format of the results: use \code{'form'} to organize results in a list of forms,
\code{'model'} to organize results in a list of models, use \code{'simple'} to organize results in
data.frame after removing item paraemters.}

\item{silent}{\code{TRUE} to mute solution information}

\item{time_limit}{the time limit in seconds passed along to solvers}

\item{message}{\code{TRUE} to print messages from solvers}
}
\value{
\code{ata} returns a \code{ata} object

\code{ata_solve} returns a solved \code{ata} object
}
\description{
\code{ata} creates a basic ATA model

\code{ata_relative_objective} adds a relative objective to the model

\code{ata_absolute_objective} adds an absolute objective to the model

\code{ata_constraint} adds a constraint to the model

\code{ata_item_use} limits the minimum and maximum usage for items

\code{ata_item_enemy} adds an enemy-item constraint to the model

\code{ata_item_fix} forces an item to be selected or not selected

\code{ata_solve} solves the MIP model
}
\details{
The ATA model stores the definitions of a MIP model. When \code{ata_solve}
is called, a real MIP object is created from the definitions.

\code{ata_obj_relative}:
when mode='max', maximize (y-tol), subject to y <= sum(x) <= y+tol;
when mode='min', minimize (y+tol), subject to y-tol <= sum(x) <= y.
When \code{negative} is \code{TRUE}, y < 0, tol > 0.
\code{coef} can be a numeric vector that has the same length with the pool,
or a variable name in the pool, or a numeric vector of theta points.
When \code{tol} is \code{NULL}, it is optimized; when it's \code{FALSE}, ignored;
when it's a number, fixed; when it's a range, constrained with lower and upper bounds.

\code{ata_obj_absolute} minimizes y0+y1 subject to t-y0 <= sum(x) <= t+y1.

When \code{level} is \code{NA}, it is assumed that the constraint is on
a quantitative item property; otherwise, a categorical item property.
\code{coef} can be a variable name, a constant, or a numeric vector that has
the same size as the pool.

\code{ata_solve} takes control options in \code{...}.
For lpsolve, see \code{lpSolveAPI::lp.control.options}.
For glpk, see \code{glpkAPI::glpkConstants}\cr
Once the model is solved, additional data are added to the model.
\code{status} shows the status of the solution, \code{optimum}
the optimal value of the objective fucntion found in the solution,
\code{obj_vars} the values of two critical variables in the objective
function, \code{result} the assembly results in a binary matrix, and
\code{items} the assembled items
}
\examples{
## generate a pool of 100 items
library(Rirt)
n_items <- 100
pool <- with(model_3pl_gendata(1, n_items), data.frame(id=1:n_items, a=a, b=b, c=c))
pool$content <- sample(1:3, n_items, replace=TRUE)
pool$time <- round(rlnorm(n_items, log(60), .2))
pool$group <- sort(sample(1:round(n_items/3), n_items, replace=TRUE))
pool <- list('3pl'=pool)

## ex. 1: four 10-item forms, maximize b parameter
x <- ata(pool, 4, test_len=10, max_use=1)
x <- ata_relative_objective(x, "b", "max")
x <- ata_solve(x, time_limit=2)
with(x$items$'3pl', aggregate(b, by=list(form=form), mean))
with(x$items$'3pl', table(form))

\donttest{
## ex. 2: four 10-item forms, minimize b parameter
x <- ata(pool, 4, test_len=10, max_use=1)
x <- ata_relative_objective(x, "b", "min", negative=TRUE)
x <- ata_solve(x, time_limit=5)
with(x$items$'3pl', aggregate(b, by=list(form=form), mean))
with(x$items$'3pl', table(form))

## ex. 3: two 10-item forms, mean(b)=0, sd(b)=1
## content = (3, 3, 4), avg. time = 55--65 seconds
constr <- data.frame(name='content',level=1:3, min=c(3,3,4), max=c(3,3,4), stringsAsFactors=FALSE)
constr <- rbind(constr, c('time', NA, 55*10, 65*10))
x <- ata(pool, 2, test_len=10, max_use=1)
x <- ata_absolute_objective(x, pool$'3pl'$b, target=0*10)
x <- ata_absolute_objective(x, (pool$'3pl'$b-0)^2, target=1*10)
for(i in 1:nrow(constr))
  x <- with(constr, ata_constraint(x, name[i], min[i], max[i], level=level[i]))
x <- ata_solve(x)
with(x$items$'3pl', aggregate(b, by=list(form=form), mean))
with(x$items$'3pl', aggregate(b, by=list(form=form), sd))
with(x$items$'3pl', aggregate(time, by=list(form=form), mean))
with(x$items$'3pl', aggregate(content, by=list(form=form), function(x) freq(x, 1:3)$freq))

## ex. 4: two 10-item forms, max TIF over (-1, 1), consider item sets
x <- ata(pool, 2, test_len=10, max_use=1, group="group")
x <- ata_relative_objective(x, seq(-1, 1, .5), 'max')
x <- ata_solve(x, time_limit=5)
plot(x)
}
}
