% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/regu-est-c.r
\name{glm.regu.cv}
\alias{glm.regu.cv}
\title{Regularied M-estimation for fitting generalized linear models based on cross validation}
\usage{
glm.regu.cv(fold, nrho = NULL, rho.seq = NULL, y, x, iw = NULL,
  loss = "cal", n.iter = 100, eps = 1e-06, tune.fac = 0.5,
  tune.cut = TRUE, ann.init = TRUE, nz.lab = NULL, permut = NULL)
}
\arguments{
\item{fold}{A fold number used for cross validation.}

\item{nrho}{The number of tuning parameters searched in cross validation.}

\item{rho.seq}{A vector of tuning parameters searched in cross validation. If both \code{nrho} and \code{rho.seq} are specified, then \code{rho.seq} overrides \code{nrho}.}

\item{y}{An \eqn{n} x \eqn{1} response vector.}

\item{x}{An \eqn{n} x \eqn{p} matix of covariates, excluding a constant.}

\item{iw}{An \eqn{n} x \eqn{1} weight vector.}

\item{loss}{A loss function, which can be specified as "gaus" for continuous responses, or "ml" or "cal" for binary respones.}

\item{n.iter}{The maximum number of iterations allowed as in \code{\link{glm.regu}}.}

\item{eps}{The tolerance used to declare convergence as in \code{\link{glm.regu}}.}

\item{tune.fac}{The multiplier (factor) used to define \code{rho.seq} if only \code{nrho} is specified.}

\item{tune.cut}{Logical; if \code{TRUE}, all smaller tuning parameters are skipped once non-convergence is found with a tuning parameter.}

\item{ann.init}{Logical; if \code{TRUE}, the estimates from the previous tuning parameter are used as the inital values when fitting with the current tuning parameter.}

\item{nz.lab}{A \eqn{p} x \eqn{1} logical vector (useful for simulations), indicating which covariates are included when calculating the number of nonzero coefficients.}

\item{permut}{An \eqn{n} x \eqn{1} vector, giving a random permutation of the integers from 1 to \eqn{n}, which is used in cross validation.}
}
\value{
\item{permut}{An \eqn{n} x \eqn{1} vector, giving the random permutation used in cross validation.}
\item{rho}{The vector of tuning parameters, searched in cross validation.}
\item{non.conv}{A vector indicating the non-convergene status found or imputed if \code{tune.cut=TRUE}, for the tuning parmaters in cross validation.
For each tuning parameter, 0 indicates convergence, 1 non-convergence if exceeding \code{n.iter}, 2 non-convergence if exceeding \code{bt.lim}.}
\item{err.ave}{A vector giving the averages of the test losses in cross validation.}
\item{err.sd}{A vector giving the standard deviations of the test losses in cross validation.}
\item{sel.rho}{A vector of two selected tuning parameters by cross validation; see \strong{Details}.}
\item{sel.nz}{A vector of numbers of nonzero coefficients estimated for the selected tuning parameters.}
\item{sel.bet}{The \eqn{(p+1)} x \eqn{2} vector of estimated intercept and coefficients.}
\item{sel.fit}{The \eqn{n} x \eqn{2} vector of fitted values.}
}
\description{
This function implements regularized M-estimation for fitting generalized linear models with binary or contiunous responses 
based on cross validation.
}
\details{
Cross validation is performed as described in Tan (2020a, 2020b). If not specified by users, the sequence of tuning parameters searched is defined as 
a geometric series of length \code{nrho}, starting from the value which yields a zero solution, and then decreasing by a factor \code{tune.fac} successively. 

After cross validation, two tuning parameters are selected. The first and default choice is the value yielding the smallest average test loss.
The second choice is the largest value giving the average test loss within one standard error of the first choice (Hastie, Tibshirani, and Friedman 2016).
}
\examples{
\donttest{
data(simu.data)
n <- dim(simu.data)[1]
p <- dim(simu.data)[2]-2

y <- simu.data[,1]
tr <- simu.data[,2]
x <- simu.data[,2+1:p]
x <- scale(x)

### Example 1: Regularized maximum likelihood estimation of propensity scores
ps.cv.rml <- glm.regu.cv(fold=5, nrho=1+10, y=tr, x=x, loss="ml")
ps.cv.rml$rho
ps.cv.rml$err.ave
ps.cv.rml$err.sd
ps.cv.rml$sel.rho
ps.cv.rml$sel.nz

fp.cv.rml <- ps.cv.rml $sel.fit[,1]
check.cv.rml <- mn.ipw(x, tr, fp.cv.rml)
check.cv.rml$est

### Example 2: Regularized calibrated estimation of propensity scores
ps.cv.rcal <- glm.regu.cv(fold=5, nrho=1+10, y=tr, x=x, loss="cal")
ps.cv.rcal$rho
ps.cv.rcal$err.ave
ps.cv.rcal$err.sd
ps.cv.rcal$sel.rho
ps.cv.rcal$sel.nz

fp.cv.rcal <- ps.cv.rcal $sel.fit[,1]

check.cv.rcal <- mn.ipw(x, tr, fp.cv.rcal)
check.cv.rcal$est
}

}
\references{
Hastie, T., Tibshirani, R., and Friedman. J. (2016) \emph{The Elements of Statistical Learning} (second edition), Springer: New York.

Tan, Z. (2020a) Regularized calibrated estimation of propensity scores with model misspecification and high-dimensional data, \emph{Biometrika}, 107, 137–158.

Tan, Z. (2020b) Model-assisted inference for treatment effects using regularized calibrated estimation with high-dimensional data, \emph{Annals of Statistics}, 48, 811–837.
}
