% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/track_intersection.R
\name{track_intersection}
\alias{track_intersection}
\title{Calculate intersection metrics in tracks}
\usage{
track_intersection(
  data,
  test = NULL,
  H1 = NULL,
  sim = NULL,
  origin.permutation = NULL,
  custom.coord = NULL
)
}
\arguments{
\item{data}{A \code{track} R object, which is a list consisting of two elements:
\itemize{
\item \strong{\code{Trajectories}}: A list of interpolated trajectories, where each trajectory is a series of midpoints between consecutive footprints.
\item \strong{\code{Footprints}}: A list of data frames containing footprint coordinates, metadata (e.g., image reference, ID), and a marker indicating whether the footprint is actual or inferred.
}}

\item{test}{Logical; if \code{TRUE}, the function compares the observed intersection metrics against. Default is \code{FALSE}.}

\item{H1}{A character string specifying the alternative hypothesis to be tested. Options are \code{"Lower"} for testing whether the observed intersections are significantly lower than the simulated ones (e.g., coordinated or gregarious movement), or \code{"Higher"} for testing whether the observed intersections are significantly higher than the simulated ones (e.g., predatory or chasing events).}

\item{sim}{A \code{track simulation} R object consisting of a list of simulated trajectories to use for comparison when \code{test = TRUE}.}

\item{origin.permutation}{A character string specifying the method for permutation of the coordinates of the simulated trajectories' origins.
Options include \code{"None"}, \code{"Min.Box"}, \code{"Conv.Hull"}, or \code{"Custom"}. Default is \code{"None"}.}

\item{custom.coord}{A matrix of custom coordinates that define the vertices of an area for permutation of the coordinates of the simulated
trajectories' origins.}
}
\value{
A \code{track intersection} R object consisting of a list containing the following elements:
\item{Intersection_metric}{A matrix of unique intersection counts between trajectories. Each entry
represents the number of unique intersection points between the corresponding pair of trajectories.}
\item{Intersection_metric_p_values}{(If \code{test = TRUE}) A matrix of \emph{p}-values associated with
the intersection metrics, calculated through permutations of simulated trajectory origins. Each entry
reflects the probability of observing an intersection count as extreme as the observed one,
given the null hypothesis of no difference.}
\item{Intersection_metric_p_values_combined}{(If \code{test = TRUE}) A numeric value representing
the combined \emph{p}-value for all intersections, indicating the overall significance of the
intersection metrics across all pairs of trajectories.}
\item{Intersection_metric_simulations}{(If \code{test = TRUE}) A list containing matrices of
intersection counts for each simulation iteration, allowing for further inspection of the
distribution of intersections across multiple randomized scenarios.}
}
\description{
\code{track_intersection()} calculates the number of unique intersections between trajectories.
The function also supports testing with simulations and different permutation procedures for the coordinates
of the simulated trajectories' origins to compute \emph{p}-values. This allows for a robust assessment of the intersection metrics,
enabling users to evaluate the significance of the observed intersections in relation to simulated trajectories.
}
\details{
The \code{track_intersection()} function is designed to calculate the number of unique intersections between trajectories
and to evaluate their statistical significance through hypothesis testing based on simulated tracks. This process provides a
robust framework for comparing observed intersections against those expected under random conditions, allowing users to test
specific behavioral hypotheses related to the movement patterns of trackmakers.

Hypothesis testing is controlled by the \code{H1} argument, which defines the \strong{alternative hypothesis} to be evaluated.
This argument is crucial for interpreting the statistical results, as it determines whether the function will test for
\strong{reduced} or \strong{increased} intersection counts compared to simulated trajectories.

The \code{H1} argument accepts two possible values:
\itemize{
\item \code{"Lower"}: This option tests whether the observed intersections are significantly lower than those generated by simulations.
This scenario corresponds to hypotheses involving \strong{coordinated or gregarious movement}, where animals moving in parallel
or in a group (e.g., hunting packs or social gatherings) would produce \strong{fewer intersections} than expected under random conditions.
\item \code{"Higher"}: This option tests whether the observed intersections are significantly higher than those generated by simulations.
It applies to scenarios where \strong{predatory or chasing interactions} are likely, such as when one trackmaker follows or crosses
the path of another. This behavior results in \strong{more intersections} than what would occur randomly.
}

The selection of the \code{H1} argument must be consistent with the behavioral hypothesis being tested. For example, use \code{"Lower"}
when investigating group movement or cooperative behavior, and \code{"Higher"} when analyzing predatory or competitive interactions.
The function will automatically adjust the calculation of \emph{p}-values to reflect the selected \code{H1}. If the argument is left
\code{NULL}, an error will be triggered, indicating that users must explicitly specify the hypothesis to be tested.

The interpretation of the \strong{combined \emph{p}-value} returned by the function is directly influenced by the choice of \code{H1}, as it determines
whether the statistical comparison aims to detect a \strong{reduction} or an \strong{increase} in intersection counts compared to the simulated dataset.

In addition to hypothesis testing, the \code{track_intersection()} function offers several options for altering the initial positions
of simulated tracks through the \code{origin.permutation} argument. The available options include:
\itemize{
\item \code{"None"}: Simulated trajectories are not shifted and are compared based on their original starting positions.
\item \code{"Min.Box"}: Trajectories are randomly placed within the \strong{minimum bounding box} surrounding the original starting points.
\item \code{"Conv.Hull"}: Trajectories are placed within the \strong{convex hull} that encompasses all original starting points, providing
a more precise representation of the area occupied by the tracks.
\item \code{"Custom"}: Allows users to define a specific region of interest by providing a matrix of coordinates (\code{custom.coord}) that
specifies the vertices of the desired area. This option is particularly useful when certain spatial features or environmental conditions
are known to constrain movement.
}

The choice of \code{origin.permutation} should reflect the nature of the behavioral hypothesis being tested. For example, using
\code{"None"} is most appropriate when testing how intersections compare under scenarios where trackmakers originate from specific
locations. In contrast, options like \code{"Min.Box"}, \code{"Conv.Hull"}, or \code{"Custom"} are suitable when evaluating how intersections
would differ if the tracks originated from a broader or predefined area.

The \code{track_intersection()} function also allows for integration with similarity metrics computed using \code{simil_DTW_metric()} and
\code{simil_Frechet_metric()}. This combination of intersection counts and similarity metrics can provide a more comprehensive analysis
of how trackmakers interacted, whether their movements were coordinated or independent, and whether their interactions were consistent
with the hypothesized behavioral patterns.

Overall, the selection of \code{H1} and \code{origin.permutation} should be carefully considered in light of the specific hypotheses
being tested. By combining intersection metrics with similarity measures, users can obtain a deeper understanding of the
behavioral dynamics underlying the observed trackways.
}
\section{Logo}{

\if{html}{\figure{Logo.png}{options: width=30\%}}
}

\examples{
# Example 1: Intersection metrics in the PaluxyRiver dataset.
s1 <- simulate_track(PaluxyRiver, nsim = 5, model = "Directed")
int1 <- track_intersection(PaluxyRiver, test = TRUE, H1 = "Lower", sim = s1,
  origin.permutation = "None")
print(int1)

# Example 2: Using "Min.Box" origin permutation in PaluxyRiver dataset.
s2 <- simulate_track(PaluxyRiver, nsim = 5, model = "Constrained")
int2 <- track_intersection(PaluxyRiver, test = TRUE, H1 = "Lower", sim = s2,
  origin.permutation = "Min.Box")
print(int2)

# Example 3: Using "Conv.Hull" origin permutation in PaluxyRiver dataset.
s3 <- simulate_track(PaluxyRiver, nsim = 5, model = "Unconstrained")
int3 <- track_intersection(PaluxyRiver, test = TRUE, H1 = "Lower", sim = s3,
  origin.permutation = "Conv.Hull")
print(int3)

# Example 4: Using "Min.Box" origin permutation in MountTom subset.
sbMountTom <- subset_track(MountTom, tracks = c(1, 2, 3, 4, 7, 8, 9, 13, 15, 16, 18))
s4 <- simulate_track(sbMountTom, nsim = 5)
int4 <- track_intersection(sbMountTom, test = TRUE, H1 = "Higher", sim = s4,
  origin.permutation = "Min.Box")
print(int4)

# Example 5: Customized origin permutation in MountTom subset.
sbMountTom <- subset_track(MountTom, tracks = c(1, 2, 3, 4, 7, 8, 9, 13, 15, 16, 18))
s5 <- simulate_track(sbMountTom, nsim = 5)
area_origin <- matrix(c(50, 5, 10, 5, 10, 20, 50, 20), ncol = 2, byrow = TRUE)
int5 <- track_intersection(sbMountTom, test = TRUE, H1 = "Higher", sim = s5,
  origin.permutation = "Custom", custom.coord = area_origin)
print(int5)

}
\seealso{
\code{\link{tps_to_track}}, \code{\link{simulate_track}}, \code{\link{simil_DTW_metric}}, \code{\link{simil_Frechet_metric}}
}
\author{
Humberto G. Ferrón

humberto.ferron@uv.es

Macroevolution and Functional Morphology Research Group (www.macrofun.es)

Cavanilles Institute of Biodiversity and Evolutionary Biology

Calle Catedrático José Beltrán Martínez, nº 2

46980 Paterna - Valencia - Spain

Phone: +34 (9635) 44477
}
