#' Generate parental allocations for a meiotic tetrad; developer function.
#'
#' For a vector of per marker chromosome indicators, generate per marker
#' parental ids for a meiotic tetrad generated by chromosomal crossovers
#' followed by independent assortment. We assume
#' \enumerate{
#' \item For all chromosomes, both pairs of non-sister chromatids cross over
#' \item One crossover per non-sister chromatid pair
#' \item Equal-length chromosomes (follows from above; in reality, chromosomes
#' have different lengths, and the number of crossovers increases with length)
#' \item Equi-distributed markers (implicit)
#' }
#'
#' @param chrs_per_marker A vector of chromosome numbers for each marker.
#'
#' @examples
#' n_chrs <- 14 # P. vivax has 14 chromosomes
#' n_markers <- 100 # For 100 markers
#' chrs_per_marker <- round(seq(0.51, n_chrs + 0.5, length.out = n_markers))
#' recombine_parent_ids(chrs_per_marker)
#'
#' @noRd
recombine_parent_ids <- function(chrs_per_marker) {

  if(!all(cummax(chrs_per_marker) == chrs_per_marker)) {
    stop("Chromosomes must be increasing order")
  }

  per_chr_marker_counts <- table(chrs_per_marker)

  chi_count <- 1 # Assumes one chiasmata per non-sister chromatid pair
  chr_count <- length(per_chr_marker_counts) # Number of chromosomes

  # Sample per chromosome parent one segment lengths (in units of markers)
  # for chromatid sets one and two independently
  c1_p1_segment_length <- sapply(per_chr_marker_counts, sample, size = chi_count)
  c2_p1_segment_length <- sapply(per_chr_marker_counts, sample, size = chi_count)

  # Parent allocation for chromatid set one after crossover
  c1 <- do.call("c", lapply(1:chr_count, function(i) {
    x <- rep(2, per_chr_marker_counts[i])
    x[1:c1_p1_segment_length[i]] <- 1
    return(x)
  }))

  # Parent allocation for chromatid set two after crossover
  c2 <- do.call("c", lapply(1:chr_count, function(i) {
    x <- rep(2, per_chr_marker_counts[i])
    x[1:c2_p1_segment_length[i]] <- 1
    return(x)
  }))

  # Generate complementary chromatid sets
  c3 <- abs(c1-2) + 1
  c4 <- abs(c2-2) + 1

  # Check complements
  if (!all(c(c1+c3, c2+c4) == 3)) stop ("recombinant chromatids not complementary")

  # Parent allocation for the tetrad pre independent orientation
  cs_pre <- cbind(c1,c2,c3,c4)

  # Independent orientation (shuffle the chromatid sets)
  recomb_chromatid_ids <- sapply(1:chr_count, function(chr) sample(x = 1:4, size = 4))

  # Parent allocation for the tetrad post independent orientation
  cs <- sapply(1:4, function(i) {
    c_ind <- recomb_chromatid_ids[i,]
    do.call("c", lapply(1:chr_count, function(j) {
      m_ind <- which(chrs_per_marker == unique(chrs_per_marker)[j])
      cs_pre[m_ind,c_ind[j]]
    }))
  })

  rownames(cs) <- NULL

  # Return per-marker parental ids for the tetrad
  return(cs)
}


