% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ProbitSpatial.R
\name{sim_binomial_probit}
\alias{sim_binomial_probit}
\title{Simulate the dependent variable of a SAR/SEM/SARAR model.}
\usage{
sim_binomial_probit(W,X,beta,rho,model="SAR",M=NULL,lambda=NULL,
sigma2=1,ord_iW=6,seed=123)
}
\arguments{
\item{W}{the spatial weight matrix (works for \code{"SAR"} and 
\code{"SEM"} models).}

\item{X}{the matrix of covariates.}

\item{beta}{the value of the covariates parameters.}

\item{rho}{the value of the spatial dependence parameter (works for 
\code{"SAR"} and \code{"SEM"} models).}

\item{model}{the type of model, between \code{"SAR"}, \code{"SEM"}, 
\code{"SARAR"} (Default is \code{"SAR"}).}

\item{M}{the second spatial weight matrix (only if \code{model} is 
\code{"SARAR"}).}

\item{lambda}{the value of the spatial dependence parameter (only if 
\code{model} is \code{"SARAR"}).}

\item{sigma2}{the variance of the error term (Defaul is 1).}

\item{ord_iW}{the order of approximation of the matrix 
\eqn{(I_n-\rho W)^{-1}}.}

\item{seed}{to set the random generator seed of the error term.}
}
\value{
a vector of zeros and ones
}
\description{
The function \code{sim_binomial_probit} is used to generate the dependent 
	variable of a spatial binomial probit model, where all the data and 
	parameters of the model can be modified by the user.
}
\details{
The \code{sim_binomial_probit} generates a vector of dependent 
	variables for a spatial probit model. It allows to simulate the following 
	DGPs (Data Generating Process):
	SAR
	\deqn{z = (I_n-\rho W)^{-1}(X\beta+\epsilon)	}
	SEM
	\deqn{z = X\beta+(I_n-\rho W)^{-1}\epsilon	}
	SARAR
	\deqn{z = (I_n-\rho W)^{-1}(X\beta+(I_n-\lambda M)^{-1}\epsilon)	}
	where \eqn{\epsilon} are independent and normally distributed with mean zero 
	and variance \code{sigma2} (default is 1).

	The matrix \code{X} of covariates, the corresponding parameters \code{beta}, 
	the spatial weight matrix \code{W} and the corresponding spatial dependence 
	parameter \code{rho} need to be passed by the user. Eventually, the same 
	applies for \code{lambda} and \code{M} for the SARAR model.
	
	The matrix \eqn{(I_n-\rho W)^{-1}} is computed using the 
	\code{ApproxiW} function, that can either invert \eqn{(I_n-\rho W)} 
	exactely, if \code{order_iW=0} (not suitable for \code{n} bigger than 1000),  
	or using the Taylor approximation 
	\deqn{(I_n-\rho W)^{-1}= I_n+\rho W+\rho^2 W^2+\ldots 	}
	of order \code{order_iW} (default is approximation of order 6).
}
\examples{
n <- 500
nneigh <- 3
rho <- 0.5
beta <- c(4,-2,1)
W <- generate_W(n,nneigh)
X <- cbind(1,rnorm(n,2,2),rnorm(n,0,1))
#SAR
y <- sim_binomial_probit(W,X,beta,rho,model="SAR") #SAR model
#SEM
y <- sim_binomial_probit(W,X,beta,rho,model="SEM") #SEM model
#SARAR
M <- generate_W(n,nneigh,seed=1)
lambda <- -0.5
y <- sim_binomial_probit(W,X,beta,rho,model="SARAR",M=M,lambda=lambda) 
}
\seealso{
\code{\link{generate_W}}, \code{\link{ProbitSpatialFit}}.
}
