% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/optim.fit.R
\name{optim.fit}
\alias{optim.fit}
\title{Fit multi-state model for optimization}
\usage{
optim.fit(data, transM, nstate, state_label, event_label, treatment = NULL,
  absorb, cov, cov_value)
}
\arguments{
\item{data}{Data frame in wide format where each row in the data corresponds to a single subject.
Time to a state and the occurrence of the state come in pairs. If a state is not occur,
use the time to an absorbing state or censoring time instead. Covariates are added at the end
of each row.}

\item{transM}{A \eqn{nstate \times nstate} matrix used to indicate the transitions in multi-state model.
If a transition exists between two states,
set 1 in a corresponding location, otherwise set 0.}

\item{nstate}{Number of states incorporated in the multi-state model.}

\item{state_label}{A character vector of length \code{nstate}  containing the names of states.
The elements in \code{state_label} are
extracted from the column names of \code{data}, except for the first one, which is a potential state
at the initiation of a study for each subject. Assume all subjects have the same initial state.}

\item{event_label}{A character vector of length \code{nstate-1}, indicating the occurrence of each state.
The first state in \code{state_label} do not need an indiator, as it always exists.}

\item{treatment}{A character vector of length 2, indicating whether there is a treatment variable
available. If true, the name and indicator of this treatment extracted from \code{state_label}
and \code{event_label} consist of \code{treatment}. If not, \code{treatment}=NULL. See details.
The default value is NULL.}

\item{absorb}{A character vector of length 2, indicating the name and indicator of the absorb state.}

\item{cov}{A character vector containing the names of covariates that have some effect to transition
probabilities.}

\item{cov_value}{A numeric vector containing the values of covariates corresponding to \code{cov}.
\code{cov_value} are used to calculated subject specified transition probabilities.}
}
\value{
If \code{treatment} is NULL, a list object called "overall" is output with elements:
\item{transMat}{A transition matrix describing the states and transitions in multi-state model.}
\item{tranProb}{A list of size \code{nstate} recording the transition probabilities form each state to  another along with standard errors. Element \code{[[s]]} is a data frame containing transition  probabilities from state \code{s} to state \code{1,2,...,nstate}.These transition probabilities are  time-varying over distinct transition time points.}
\item{coxobj}{An object returned by \code{coxph()} function in \code{survival} package.}
\item{ntrans}{The number of available transitions among multiple states.}
\item{...}{Other variables that extracted from the original input.}
If \code{treatment} is not NULL, three list objects called "overall", "treat","no_treat" are output.
A list "overall" contains elements:
\item{transMat}{A transition matrix describing the states and transitions in multi-state model.}
\item{ntrans}{The number of available transitions among multiple states.}
\item{...}{Other variables that extracted from the original input.}
A list "no_treat" contains elements by fitting a model for pre-treatment states:
\item{transMat}{A transition matrix describing the states and transitions if the new treatment is  not carried out.}
\item{tranProb}{A list recording the transition probabilities among pre-treatment states.}
\item{coxobj}{An object returned by \code{coxph()} function in \code{survival} package.}
\item{ntrans}{The number of available transitions among pre-treatment states.}
\item{data}{A data set contaning states if the new treatment is not carried out.}
\item{nstate}{The number of pre-treatment states.}
\item{...}{Other variables that extracted from the original input.}
A list "treat" contains elements by fitting a model if a new treatment is carried out:
\item{transMat}{A transition matrix describing the states and transitions if the new treatment is  carried out.}
\item{tranProb}{A list recording the transition probabilities among post-treatment states.}
\item{coxobj}{An object returned by \code{coxph()} function in \code{survival} package.}
\item{ntrans}{The number of available transitions among post-treatment states.}
\item{data}{A data set contaning states if the new treatment is carried out.}
\item{nstate}{The number of post-treatment states.}
\item{...}{Other variables that extracted from the original input.}
}
\description{
This function produces transition probabilities for given covariates values in multi-state models.
}
\details{
For optim.fit, transition probabilities are estimated under Markov assumption, which implies that
the probability of transition to a future state depends only on the present state, not on the history.
Taking covariates at baseline into consideration, transition probabilities can be subject-specific.
Cox proportional hazards model is used to fit transition hazards among multiple states by assuming
each transition has its own baseline hazard, and covariates have different effects on different transitions.

Let \eqn{\mathbf{S}={1,2,\cdots,S}} denote the states in the multi-state model
 and \eqn{X(t)} be a random process taking values from \eqn{\mathbf{S}}.
Denote \eqn{\alpha_{gh}(t)} as hazard ratio or transition intensity and \eqn{Z} as baseline covariates.
The instantaneous risk of a transition from state \eqn{g} into state \eqn{h} at time \eqn{t} can be
fitted by semi-parametric Cox model:
\deqn{\alpha_{gh}(t|Z)=\alpha_{gh,0}exp(\beta^{T}Z_{gh}).}
The cumulative hazard ratio is defined as \eqn{A_{gh}(t)=\int_0^t \alpha_{gh}(u)du}.
Primary interest in this function is to estimate transition probability \eqn{P_{gh}(s,t)=P(X(t)=h|X(s)=g)},
indicating the chance of transition from state \eqn{g} at time \eqn{s} to state \eqn{h} at time \eqn{t}.
Written in matrix form, transition probability matrix \eqn{\mathbf{P}(t)} can be calculated by
means of a product integral: \eqn{\mathbf{P}(s,t)=\prod_{(s,t]}(\mathbf{I}+d\mathbf{A}(u))},
where \eqn{\mathbf{A}(t)} is a transition intensity matrix. Both \eqn{\mathbf{P}} and \eqn{\mathbf{A}}
are \eqn{S \times S} matrix.

The data format required by this function is wide format, which can be regarded as the augmented
data used in single event survival analysis. For example, if there is a "recurrence" state in a
multi-state model, two variable are needed to describe this event, namely, "rec" and "rec.s".
The former is a time variable, indicating the time from initiation of the study
to the occurrence of this state, while the latter is an indicator variable with 1 for occurrence and 0 for
censoring.
If the event is censored for some patients, use the maximum follow-up instead of the event time.
Other states are prepared in the same way. Thus, each row in the augmented data
summarize all possible events for a single subject. For covariates, they are located at the end
of each row.

If the time of new treatment initiation is provided in data, the argument \code{treatment}
 should be assigned as, eg.\code{treatment=c("sct","sct.s")}. Additionally, the argument
 \code{state_lable} and \code{event_label} shoud be arranged in such order: pre-treatment state,
 treatment state, post-treatment states and  absorbing state. Assume treatment may take place
 at any pre-treatment states.  In this case,
 \code{optim.fit} function automatically fit two multistate models, one for post-treatment states
  if a new treatment is carried out , and
the other  for pre-treatment states if a new treatment is not carried out.
Thus, comparison among strategies of whether and when to initiate new treatment can be
performed in \code{sim.MTL} function.   If \code{treatment=NULL},
a single multistate model will be fitted.
}
\examples{

\dontrun{
library(OptimalTiming)
## read data
data(SimCml)

## fit multistate model if the time to new treatment initiation is available in SimCml
fit=optim.fit(data=SimCml,
         transM=matrix(c(0,1,0,0,0,1,0,0,0,1,0,1,1,1,0,0,0,1,1,1,1,
         0,0,0,0,1,1,1,0,0,0,0,0,1,1,0,0,0,0,0,0,1,0,0,0,0,0,0,0),7,byrow=TRUE),
         nstate=7,state_label=c("diagnose","cp1","ap","cp2","bc","sct","death"),
         event_label=c("cp1.s","ap.s","cp2.s","bc.s","sct.s","death.s"),
         treatment=c("sct","sct.s"),absorb=c("death","death.s"),
         cov=c("age"),cov_value=c(0))

## view the content of this object
objects(fit)

## output transition probabilities
fit$treat$tranProb
fit$no_treat$tranProb}

\dontshow{
library(OptimalTiming)
data(SimCml)
SimCml=SimCml[SimCml$sct.s==0,]
SimCml=SimCml[,c(1,2,3,4,11,12,13,14)]
fit=optim.fit(data=SimCml,
         transM=matrix(c(0,1,1,1,0,0,1,1,0,0,0,1,0,0,0,0),4,byrow=TRUE),
         nstate=4,state_label=c("diagnose","cp1","ap","death"),
         event_label=c("cp1.s","ap.s","death.s"),
         absorb=c("death","death.s"),
         cov=c("age"),cov_value=c(0))


data(SimCml)
SimCml=SimCml[,c(1,2,9,10,11,12,13,14)]
fit=optim.fit(data=SimCml,
         transM=matrix(c(0,1,1,1,0,0,1,1,0,0,0,1,0,0,0,0),4,byrow=TRUE),
         nstate=4,state_label=c("diagnose","cp1","sct","death"),
         event_label=c("cp1.s","sct.s","death.s"),
         treatment=c("sct","sct.s"),absorb=c("death","death.s"),
         cov=c("age","sex"),cov_value=c(0,1))}
}
\references{
de Wreede LC, Fiocco M, and Putter H (2010). The mstate package for estimation and prediction in non- and semi-parametric multi-state and competing risks models. Computer Methods and Programs in Biomedicine 99, 261–274.
}
