#' @title Import from previous generations of Oregon RFID antenna readers 
#' @description Only data collected using Oregon RFID (radio-frequency identification) ORMR (Oregon RFID Multi-Reader) and ORSR (Oregon RFID Single Reader) antenna readers can be imported using \code{\link{import_ORFID}}. Data from previous generations of readers must be imported using \code{\link{import_old_readers}}. Only detections are retained during compilation (events are removed).
#' 
#' This function will only work with raw data downloaded directly from Oregon RFID stationary readers. The files must be delimited (tab, comma, or semicolon) and unedited by the user.
#' @param file .txt file generated by an Oregon RFID reader.
#' @param delim field/column delimiter, which must be tab ('\\t'), comma (',') or semi-colon (';').
#' @param verbose If \emph{TRUE}, a description of the compiled data is printed to the console.
#' @details The field/column delimiter must be tab, comma or semi-colon for data compilation and further analysis. The function cannot be used for space delimited data. 
#' 
#' Data compiled using \code{\link{import_ORFID}} and \code{\link{import_old_readers}} can be joined together using \code{\link{join_multireader_data}}.
#' 
#' Note that corruption may occur in reader data files. Check your data files and compiled data carefully to ensure accuracy.
#' @return Returns a tibble object.
#' @author Hugo Marques <biohmarques@@gmail.com>
#' @importFrom magrittr %>%
#' @export
#' @examples
#'
#' # Import a single comma-deliminated data file from an old ORFID reader
#' import_old_readers(file = system.file("extdata/orfid_old_data_file.txt", 
#' package = "ORFID"), delim = ",")
#' 

import_old_readers <- function (file, delim, verbose = FALSE) {
    
    if (!(delim %in% c("\t", ",", ";"))) {
        stop("Column delimiter must be tab, comma, or semicolon")
    }
    
    if (is.na(grep("Gap", suppressWarnings(readLines(file)))[1]) == T) {
        stop("Header is missing from input file.")
    }
    
    raw_data <- suppressWarnings(
        readr::read_delim(file, 
                          delim = delim, 
                          skip = grep("Gap", readLines(file))[1] - 1,
                          show_col_types = FALSE))
    
    colnames(raw_data) <- c("DTY", "Date", "Time", "DUR", "Type", "TAG", "NCD", "EMP")
    
    site_name <- grep("^Reader*", suppressWarnings(readLines(file)), value = TRUE)
    
    site_code <- stringr::str_extract(site_name, '\\w*$')[1]
    
    raw_data2 <- raw_data %>% 
        dplyr::filter(DTY == "D") %>% 
        tidyr::unite("ARR", Date:Time, sep = " ", remove = TRUE) %>% 
        dplyr::mutate(ARR = readr::parse_datetime(ARR, "%Y-%m-%d %H:%M:%OS")) %>% 
        dplyr::mutate(DUR = readr::parse_time(DUR, "%H:%M:%OS")) %>%
        tidyr::separate(Type, sep = 1, into = c("TCH", "TTY")) %>% 
        dplyr::mutate(TTY = as.factor(TTY)) %>% 
        dplyr::mutate(SCD = as.factor(site_code)) %>%
        dplyr::distinct()
    
    raw_data2 <- raw_data2 %>% 
        suppressWarnings(dplyr::mutate(EMP = as.double(EMP)))
    
    if (verbose == TRUE) {
        return(dplyr::glimpse(raw_data2))
    } else {
        return(raw_data2)
    }
}
