% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simConnectivity.R
\name{modelCountDataJAGS}
\alias{modelCountDataJAGS}
\title{Estimates population-level relative abundance from count data}
\usage{
modelCountDataJAGS(count_data, ni = 20000, nt = 5, nb = 5000, nc = 3)
}
\arguments{
\item{count_data}{List containing the following elements:
' \describe{
   \item{\code{C}}{nYears by nRoutes matrix containing the observed number of individuals counted at each route in each year.}
   \item{\code{strat}}{Vector of length nRoutes indicating the population/region in which each route is located.}
   \item{\code{routesPerStrata}}{Vector of length 1 or nStrata containing the number of routes (i.e. counts) per population. If length(routesPerStrata) == 1, number of routes is identical for each population.}
}}

\item{ni}{Number of MCMC iterations. Default = 20000.}

\item{nt}{Thinning rate. Default = 5.}

\item{nb}{Number of MCMC iterations to discard as burn-in. Default = 5000.}

\item{nc}{Number of chains. Default = 3.}
}
\value{
\code{modelCountDataJAGS} returns an mcmc object containing posterior samples for each monitored parameter.
}
\description{
Uses a Bayesian hierarchical model to estimate relative abundance of regional
populations from count-based data (e.g., Breeding Bird Survey)
}
\examples{
set.seed(150)

### Set parameters for simulation ----

# Number of populations
nStrata. <- 4
# Number of routes w/i each population (assumed to be balanced)
routePerStrat. <- 30 # reduced from 90 for example speed
# Number of years
nYears. <- 5 # reduced from 10 for example speed
# log(Expected number of birds counted at each route)
alphaStrat. <- 1.95
# standard deviation of normal distribution assumed for route/observer random
# effects
sdRoute. <- 0.6
# standard deviation of normal distribution assumed for year random effects
sdYear. <- 0.18


# Number of simulated datasets to create and model
nsims <- 50 # reduced from 100 for example speed
# Number of MCMC iterations
ni. <- 1000 # reduced from 15000 for example speed
# Number of iterations to thin from posterior (reduced from 5)
nt. <- 1
# Number of iterations to discard as burn-in
nb. <- 500 # reduced from 5000 for example speed
# Number of MCMC chains
nc. <- 1 # reduced from 3 for example speed

### Create empty matrix to store model output ---
sim_in <- vector("list", nsims)
sim_out <- vector("list", nsims)


# Simulation ---
\donttest{
system.time(for(s in 1:nsims){
  cat("Simulation",s,"of",nsims,"\n")

  # Simulate data
  sim_data <- simCountData(nStrata = nStrata., routesPerStrata = routePerStrat.,
                           nYears = nYears., alphaStrat = alphaStrat.,
                           sdRoute = sdRoute., sdYear = sdYear.)
  sim_in[[s]] <- sim_data


  # Estimate population-level abundance
  out_mcmc <- modelCountDataJAGS(count_data = sim_data, ni = ni., nt = nt.,
                                 nb = nb., nc = nc.)

  # Store model output
  sim_out[[s]] <- out_mcmc
  remove(out_mcmc)

})


### Check that relative abundance is, on average, equal for each population
prop.table(sapply(sim_in, function(x) return(rowsum(colSums(x$C), x$strat))), 2)

rel_names <- paste0('relN[', 1:nStrata., ']')
rel_abund1 <- data.frame(sim=1:nsims,
                         ra1.mean=NA, ra2.mean=NA, ra3.mean=NA, ra4.mean=NA,
                         ra1.low=NA, ra2.low=NA, ra3.low=NA, ra4.low=NA,
                         ra1.high=NA, ra2.high=NA, ra3.high=NA, ra4.high=NA,
                         ra1.cover=0, ra2.cover=0, ra3.cover=0, ra4.cover=0)
for (s in 1:nsims) {
  rel_abund1[s, 2:5] <- summary(sim_out[[s]])$statistics[rel_names, "Mean"]
  rel_abund1[s, 6:9] <- summary(sim_out[[s]])$quantiles[rel_names, 1]
  rel_abund1[s, 10:13] <- summary(sim_out[[s]])$quantiles[rel_names, 5]
}
rel_abund1 <- transform(rel_abund1,
                        ra1.cover = (ra1.low<=0.25 & ra1.high>=0.25),
                        ra2.cover = (ra2.low<=0.25 & ra2.high>=0.25),
                        ra3.cover = (ra3.low<=0.25 & ra3.high>=0.25),
                        ra4.cover = (ra4.low<=0.25 & ra4.high>=0.25))

summary(rel_abund1)
}

}
\references{
Cohen, E. B., J. A. Hostetler, M. T. Hallworth, C. S. Rushing, T. S. Sillett,
and P. P. Marra. 2018. Quantifying the strength of migratory connectivity.
Methods in Ecology and Evolution 9: 513-524.
\doi{10.1111/2041-210X.12916}

Link, W. A. and J. R. Sauer. 2002. A hierarchical analysis of population
change with application to Cerulean Warblers. Ecology 83: 2832-2840.
\doi{10.1890/0012-9658(2002)083[2832:AHAOPC]2.0.CO;2}
}
