% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dcee.R
\name{dcee}
\alias{dcee}
\title{Distal Causal Excursion Effect (DCEE) Estimation}
\usage{
dcee(
  data,
  id,
  outcome,
  treatment,
  rand_prob,
  moderator_formula,
  control_formula,
  availability = NULL,
  control_reg_method = c("gam", "lm", "rf", "ranger", "sl", "sl.user-specified-library",
    "set_to_zero"),
  cross_fit = FALSE,
  cf_fold = 10,
  weighting_function = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{data}{A data.frame in long format.}

\item{id}{Character scalar: column name for subject identifier.}

\item{outcome}{Character scalar: column name for proximal/distal outcome.}

\item{treatment}{Character scalar: column name for binary treatment \{0,1\}.}

\item{rand_prob}{Character scalar: column name for randomization probability
giving \eqn{P(A_t=1\mid H_t)} (must lie in (0,1)).}

\item{moderator_formula}{RHS-only formula of moderators of the excursion effect
(e.g., `~ 1`, `~ Z`, or `~ Z1 + Z2`).}

\item{control_formula}{RHS-only formula of covariates for learning nuisance
outcome regressions. When `control_reg_method = "gam"`, `s(x)` terms are
allowed (e.g., `~ x1 + s(x2)`). For SuperLearner methods, variables are
extracted from this formula to build the design matrix `X`.}

\item{availability}{Optional character scalar: column name for availability
indicator (0/1). If `NULL`, availability is taken as 1 for all rows.}

\item{control_reg_method}{One of `"gam"`, `"lm"`, `"rf"`, `"ranger"`,
`"sl"`, `"sl.user-specified-library"`, `"set_to_zero"`. See Details.}

\item{cross_fit}{Logical; if `TRUE`, perform K-fold cross-fitting by subject id.}

\item{cf_fold}{Integer; number of folds if `cross_fit = TRUE` (default 10).}

\item{weighting_function}{Either a single numeric constant applied to all
rows, or a character column name in `data` giving decision-point weights
\eqn{\omega_t}.}

\item{verbose}{Logical; print minimal preprocessing messages (default `TRUE`).}

\item{...}{Additional arguments passed through to the chosen learner
(e.g., `num.trees`, `mtry` for random forests; `sl.library` when
`control_reg_method = "sl.user-specified-library"`).}
}
\value{
An object of class `"dcee_fit"` with components:
\describe{
  \item{call}{The matched call to \code{dcee()}.}

  \item{fit}{A list returned by the two–stage helper with elements:
    \describe{
      \item{\code{beta_hat}}{Named numeric vector of distal causal excursion
        effect estimates \eqn{\beta}. Names are \code{"Intercept"} and the
        moderator names (if any) from \code{moderator_formula}.}
      \item{\code{beta_se}}{Named numeric vector of standard errors for
        \code{beta_hat} (same order/names).}
      \item{\code{beta_varcov}}{Variance–covariance matrix of \code{beta_hat}
        (square matrix; row/column names match \code{names(beta_hat)}).}
      \item{\code{conf_int}}{Matrix of large-sample (normal) Wald
        95\% confidence intervals for \code{beta_hat};
        columns are \code{"2.5 \%"} and \code{"97.5 \%"}.}
      \item{\code{conf_int_tquantile}}{Matrix of small-sample
        (t-quantile) 95\% confidence intervals for \code{beta_hat};
        columns are \code{"2.5 \%"} and \code{"97.5 \%"}; degrees of freedom
        are provided in \code{$df} of the \code{"dcee_fit"} object.}
      \item{\code{regfit_a0}}{Stage-1 nuisance regression fit for
        \eqn{\mu_0(H_t)} (outcome model among \code{A=0}), or \code{NULL}
        when \code{control_reg_method = "set_to_zero"}. \strong{Note:} when
        \code{cross_fit = TRUE}, this is the learner object from the
        \emph{last} fold and is provided for inspection only (do not use for
        out-of-fold prediction).}
      \item{\code{regfit_a1}}{Stage-1 nuisance regression fit for
        \eqn{\mu_1(H_t)} (outcome model among \code{A=1}); same caveats as
        \code{regfit_a0} regarding \code{cross_fit}.}
    }
  }

  \item{df}{Small-sample degrees of freedom used for t-based intervals:
    number of unique subjects minus \code{length(fit$beta_hat)}.}
}
}
\description{
Fits distal causal excursion effects in micro-randomized trials using a
**two-stage** estimator: (i) learn nuisance outcome regressions
\eqn{\mu_a(H_t)} with a specified learner (parametric/ML), optionally with
cross-fitting; (ii) solve estimating equations for the distal excursion
effect parameters (\eqn{\beta}).

This wrapper standardizes inputs and delegates computation to
[dcee_helper_2stage_estimation()].
}
\details{
**Learners.**
- `gam` uses \pkg{mgcv} and supports `s(.)` terms in `control_formula`.
- `lm` uses base \code{stats::lm}.
- `rf` uses \pkg{randomForest}; `ranger` uses \pkg{ranger}.
- `sl` / `sl.user-specified-library` use \pkg{SuperLearner}. For the former,
  `sl.library = c("SL.mean", "SL.glm", "SL.earth")` are used. For the latter,
  please provide `sl.library = c("SL.mean", ...)` via `...`.

**Notes.**
- Treatment must be coded 0/1; `rand_prob` must lie strictly in (0,1).
- `control_formula = ~ 1` is only valid with `control_reg_method = "set_to_zero"`.
}
\examples{
data(data_distal_continuous, package = "MRTAnalysis")

## Fast example: marginal effect with linear nuisance (CRAN-friendly)
fit_lm <- dcee(
    data = data_distal_continuous,
    id = "userid", outcome = "Y", treatment = "A", rand_prob = "prob_A",
    moderator_formula = ~1, # marginal (no moderators)
    control_formula = ~X, # simple linear nuisance
    availability = "avail",
    control_reg_method = "lm",
    cross_fit = FALSE
)
summary(fit_lm)
summary(fit_lm, show_control_fit = TRUE) # show Stage-1 fit info

## Moderated effect with GAM nuisance (allows smooth terms); may be slower
\donttest{
fit_gam <- dcee(
    data = data_distal_continuous,
    id = "userid", outcome = "Y", treatment = "A", rand_prob = "prob_A",
    moderator_formula = ~Z, # test moderation by Z
    control_formula = ~ s(X) + Z, # smooth in nuisance via mgcv::gam
    availability = "avail",
    control_reg_method = "gam",
    cross_fit = TRUE, cf_fold = 5
)
summary(fit_gam, lincomb = c(0, 1)) # linear combo: the Z coefficient
summary(fit_gam, show_control_fit = TRUE) # show Stage-1 fit info
}

## Optional: SuperLearner (runs only if installed)
\donttest{
library(SuperLearner)
fit_sl <- dcee(
    data = data_distal_continuous,
    id = "userid", outcome = "Y", treatment = "A", rand_prob = "prob_A",
    moderator_formula = ~1,
    control_formula = ~ X + Z,
    availability = "avail",
    control_reg_method = "sl",
    cross_fit = FALSE
)
summary(fit_sl)
}
}
