% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/arriaga.R
\name{arriaga}
\alias{arriaga}
\title{classic Arriaga decomposition}
\usage{
arriaga(
  mx1,
  mx2,
  age = 0:(length(mx1) - 1),
  nx = rep(1, length(mx1)),
  sex1 = "t",
  sex2 = sex1,
  closeout = TRUE
)
}
\arguments{
\item{mx1}{numeric vector of the mortality rates (central death rates) for population 1}

\item{mx2}{numeric vector of the mortality rates (central death rates) for population 2}

\item{age}{integer vector of the lower bound of each age group (currently only single ages supported)}

\item{nx}{integer vector of age intervals, default 1.}

\item{sex1}{character either the sex for population 1: Male (\code{"m"}), Female (\code{"f"}), or Total (\code{"t"})}

\item{sex2}{character either the sex for population 2: Male (\code{"m"}), Female (\code{"f"}), or Total (\code{"t"}) assumed same as \code{sex1} unless otherwise specified.}

\item{closeout}{logical. Default \code{TRUE}. Shall we use the HMD Method Protocol to close out the \code{ax} and \code{qx} values? See details.}
}
\value{
\code{cc} numeric vector with one element per age group, and which sums to the total difference in life expectancy between population 1 and 2.
}
\description{
Following the notation given in Preston et al (2000), Arriaga's decomposition method can written as:
\deqn{_{n}\Delta_{x} = \frac{l_x^1}{l_0^1}\cdot \left( \frac{_{n}L_{x}^{2}}{l_{x}^{2}} - \frac{_{n}L_{x}^{1}}{l_{x}^{1}}\right) + \frac{T^{2}_{x+n}}{l_{0}^{1}} \cdot \left( \frac{l_{x}^{1}}{l_{x}^{2}} - \frac{l_{x+n}^{1}}{l_{x+n}^{2}}  \right) }
}
\details{
A little-known property of this decomposition method is that it is directional, in the sense that we are comparing a movement of \code{mx1} to \code{mx2}, and this is not exactly symmetrical with a comparison of \code{mx2} with \code{mx1}. Note also, if decomposing in reverse from the usual, you may need a slight adjustment to the closeout value in order to match sums properly (see examples for a demonstration).

setting \code{closeout} to \code{TRUE} will result in value of \code{1/mx} for the final age group, of \code{ax} and a value of 1 for the closeout of \code{qx}.
}
\examples{
a <- .001
b <- .07
x <- 0:100
mx1 <- a * exp(x * b)
mx2 <- a/2 * exp(x * b)
cc <- arriaga(mx1, mx2, age = x)
e01 <- mx_to_e0(mx1, age = x)
e02 <- mx_to_e0(mx2, age = x)
(delta <- e02 - e01)
sum(cc)

\donttest{
 plot(x, cc)
}
# asymmetrical with a decomposition in the opposite direction
cc2 <- -arriaga(mx1 = mx2, mx2 = mx1, age = x)
plot(x, cc)
lines(x,cc2)
# also we lose some precision?
sum(cc);sum(cc2)
# found it!
delta-sum(cc2); cc2[length(cc2)] / 2

# But this is no problem if closeout = FALSE
-arriaga(mx1 = mx2, mx2 = mx1, age = x,closeout=FALSE) |> sum()
arriaga(mx1 = mx1, mx2 = mx2, age = x,closeout=FALSE) |> sum()
}
\references{
\insertRef{arriaga1984measuring}{LEdecomp}
\insertRef{preston2000demography}{LEdecomp}
}
