% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{Soil_HBV}
\alias{Soil_HBV}
\title{Empirical soil moisture routine}
\usage{
Soil_HBV(
       model,
       inputData,
       initCond,
       param
       )
}
\arguments{
\item{model}{numeric integer suggesting one of the following options:
\itemize{
  \item 1: Classical HBV soil moisture routine.
  \item 2: HBV soil moisture routine with varying area. This option should
  be used with \code{\link{SnowGlacier_HBV}}'s \strong{\emph{model 3}}.
}}

\item{inputData}{numeric matrix with the following series

\strong{Model 1}
\itemize{
  \item \code{column_1}: \code{Total = Prain + Msnow} \eqn{[mm/\Delta t]}. This
  series comes from the output of the \code{\link{SnowGlacier_HBV}} module.
  \item \code{column_2}: potential evapotranspiration  \eqn{[mm/\Delta t]}. Since
  the package has a simple model (\code{\link{PET}}) to obtain this
  series I strongly recommend using the
  \href{https://CRAN.R-project.org/package=Evapotranspiration}{\code{Evapotranspiration}}
  package.
}

\strong{Model 2}
\itemize{
  \item \code{column_1}: as in \strong{model 1}.
  \item \code{column_2}: as in \strong{model 1}.
  \item \code{column_3} : relative soil area (ratio of soil surface over
  basin area). When the glacier area changes the soil does the same, so coherence
  between this two series should be seek.This value is used to scale the effective
  runoff accordingly (\code{Rech} column in the matrix output).
}}

\item{initCond}{numeric vector with the following values:
 \enumerate{
  \item initial soil water content \eqn{[mm]}. This is a model state variable
  and is internally used as first soil moisture value.
  \item relative area \eqn{[-]}. Only needed when using \strong{\code{model 1}}.
  This is the soil surface proportion relative to the catchment as a whole, so
  the values should never supersede one (1). This value is used to scale the effective
  runoff accordingly (\code{Rech} column in the matrix output).
}}

\item{param}{numeric vector with the following values:
\enumerate{
  \item \code{FC}: fictitious soil field capacity \eqn{[mm]}.
  \item \code{LP}: parameter to get actual ET \eqn{[-]}.
  \item \eqn{\beta}: exponential value that allows for non-linear relations between
  soil box water input (rainfall plus snowmelt) and the effective runoff \eqn{[-]}.
}}
}
\value{
Numeric matrix with the following columns:
\enumerate{
  \item \code{Rech}: recharge series \eqn{[mm/\Delta t]}. This is the input to
  the \code{\link{Routing_HBV}} module.
  \item \code{Eact}: actual evapotranspiration series \eqn{[mm/\Delta t]}.
  \item \code{SM}: soil moisture series \eqn{[mm/\Delta t]}.
}
}
\description{
This module allows you to account for actual evapotranspiration,
abstractions, antecedent conditions and effective runoff. The formulation enables
non linear relationships between soil box water input (rainfall plus snowmelt) and
the effective runoff. This effective value is the input series to the routine function
(\code{\link{Routing_HBV}}).
}
\examples{
# The following is a toy example. I strongly recommend to see
# the package vignettes in order to improve your skills on HBV.IANIGLA

# HBV soil routine with variable area
## Calder's model
potEvap <- PET(model = 1, hemis = 1, inputData = as.matrix(1:315), elev = c(1000, 1500),
              param = c(4, 0.5))

## Debris-covered ice
 ObsTemp   <- sin(x = seq(0, 10*pi, 0.1))
 ObsPrecip <- runif(n = 315, max = 50, min = 0)
 ObsGCA    <- seq(1, 0.8, -0.2/314)

## Fine debris covered layer assumed. Note that the ice-melt factor is cumpulsory but harmless.
DebrisCovGlac <- SnowGlacier_HBV(model = 3, inputData = cbind(ObsTemp, ObsPrecip, ObsGCA),
                                 initCond = c(10, 3, 1), param = c(1, 1, 0, 3, 1, 6))

## Soil routine
ObsSoCA     <- 1 - ObsGCA
inputMatrix <- cbind(DebrisCovGlac[ , 9], potEvap, ObsSoCA)

soil <- Soil_HBV(model = 2, inputData = inputMatrix, initCond = c(50), param = c(200, 0.5, 2))

}
\references{
Bergström, S., Lindström, G., 2015. Interpretation of runoff processes in hydrological
modelling—experience from the HBV approach. Hydrol. Process. 29, 3535–3545.
https://doi.org/10.1002/hyp.10510
}
