#' Variable selection for exogenous covariates in GARCHX models
#'
#' Performs variable selection on the exogenous covariates through testing each covariate in X and correcting the p-values for multiple testing.
#'
#' @param eps Time series data
#' @param X Matrix with exogenous covariates where the number of rows is equal to the length of eps
#' @param order Order of the GARCH model. Value of p cannot be 0.
#' @param delta Value of the power of the main time series to allow for Power GARCHX, default is 2 for GARCHX
#' @param alpha.level Alpha level for p-value cut-off in variable selection
#' @param adjust.method Multiple testing p-value adjustment, see p.adjust. Possible values are "holm", "hochberg", "hommel", "bonferonni", "BH", "BY", "fdr", "none"
#' @param optim.method Optimization method for maximizing quasi-likelihood function. Options: "NR", "L-BFGS-B", "GA", "PS", "SA". Default value is "NR"
#'
#' @details Using the GARCHX model \deqn{\mathcal{E}_t = \sigma_tw_t}
#' \deqn{\sigma^2_t = \omega_0 + \sum^{p}_{i=1}\alpha_i\mathcal{E}_{t-i}^2 + \sum^q_{j=1}\beta_j\sigma^2_{t-j}+\mathbf{\pi}^T\mathbf{x}_{t-1}}
#'
#' performs variable selection by testing \deqn{H_0: \pi_j = 0, \forall j}
#' and compares the p-values to the adjusted alpha level according to adjust.method.
#' If alpha.level = 1, then no variable selection is performed and the function only estimates the parameters
#'
#'
#' @return An object of class GARCHX
#'
#' @export
#'
#' @examples set.seed(123)
#' @examples pi <- c(1, 0, 0, 4)
#' @examples n <- 2000
#' @examples d <- length(pi)
#' @examples valinit <- 100
#' @examples n2 <- n + d + 1
#' @examples omega <- 0.1
#' @examples alpha <- 0.2
#' @examples beta <- 0.3
#' @examples delta <- 2
#' @examples e<-rnorm(n2+valinit)
#' @examples Y<-e
#' @examples for (t in 2:n2)
#' @examples  Y[t]<- 0.2*Y[t-1]+e[t]
#' @examples x<-exp(Y)
#' @examples X <- matrix(0, nrow = (n+valinit), ncol = length(pi))
#' @examples for(j in 1:d)
#' @examples  X[, j] <- x[(d+2-j):(n+d+1-j+valinit)]
#' @examples data <- simulate(n, omega, alpha, beta, delta, X, pi, valinit = valinit)
#' @examples model <- GARCHX_select(eps = data$eps, X = data$X)
#'
#' @references
#' Francq, C. and Thieu, L.Q.(2018). *QML Inference for Volatility Models with Covariates*.
#' Econometric Theory, Cambridge University Press
#'
GARCHX_select <- function(eps, X, order = c(1, 1), delta = 2, alpha.level = 0.05, adjust.method = "fdr", optim.method = "NR"){
  # High level/Interface function for GARCHX_select.default function
  # Calling GARCHX.select.default function
  UseMethod("GARCHX_select")
} # End GARCHXselect
