% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fnn.cv.R
\name{fnn.cv}
\alias{fnn.cv}
\title{Functional Neural Networks with Cross-validation}
\usage{
fnn.cv(
  nfolds,
  resp,
  func_cov,
  scalar_cov = NULL,
  basis_choice = c("fourier"),
  num_basis = c(7),
  hidden_layers = 2,
  neurons_per_layer = c(64, 64),
  activations_in_layers = c("sigmoid", "linear"),
  domain_range = list(c(0, 1)),
  epochs = 100,
  loss_choice = "mse",
  metric_choice = list("mean_squared_error"),
  val_split = 0.2,
  learn_rate = 0.001,
  patience_param = 15,
  early_stopping = TRUE,
  print_info = TRUE,
  batch_size = 32,
  decay_rate = 0,
  func_resp_method = 1,
  covariate_scaling = TRUE,
  raw_data = FALSE
)
}
\arguments{
\item{nfolds}{The number of folds to be used in the cross-validation process.}

\item{resp}{For scalar responses, this is a vector of the observed dependent variable. For functional responses,
this is a matrix where each row contains the basis coefficients defining the functional response (for each observation).}

\item{func_cov}{The form of this depends on whether the \code{raw_data} argument is true or not. If true, then this is
a list of k matrices. The dimensionality of the matrices should be the same (n x p) where n is the number of
observations and p is the number of longitudinal observations. If \code{raw_data} is false, then the input should be a tensor
with dimensionality b x n x k where b is the number of basis functions used to define the functional covariates, n is
the number of observations, and k is the number of functional covariates.}

\item{scalar_cov}{A matrix contained the multivariate information associated with the data set. This is all of your
non-longitudinal data.}

\item{basis_choice}{A vector of size k (the number of functional covariates) with either "fourier" or "bspline" as the inputs.
This is the choice for the basis functions used for the functional weight expansion. If you only specify one, with k > 1,
then the argument will repeat that choice for all k functional covariates.}

\item{num_basis}{A vector of size k defining the number of basis functions to be used in the basis expansion. Must be odd
for \code{fourier} basis choices. If you only specify one, with k > 1, then the argument will repeat that choice for all
k functional covariates.}

\item{hidden_layers}{The number of hidden layers to be used in the neural network.}

\item{neurons_per_layer}{Vector of size = \code{hidden_layers}. The u-th element of the vector corresponds to the number of neurons
in the u-th hidden layer.}

\item{activations_in_layers}{Vector of size = \code{hidden_layers}. The u-th element of the vector corresponds to the
activation choice in the u-th hidden layer.}

\item{domain_range}{List of size k. Each element of the list is a 2-dimensional vector containing the upper and lower
bounds of the k-th functional weight.}

\item{epochs}{The number of training iterations.}

\item{loss_choice}{This parameter defines the loss function used in the learning process.}

\item{metric_choice}{This parameter defines the printed out error metric.}

\item{val_split}{A parameter that decides the percentage split of the inputted data set.}

\item{learn_rate}{Hyperparameter that defines how quickly you move in the direction of the gradient.}

\item{patience_param}{A keras parameter that decides how many additional \code{epochs} are eclipsed with minimal change in
error before the learning process is stopped. This is only active if \code{early_stopping = TRUE}}

\item{early_stopping}{If TRUE, then learning process will be halted early if error improvement isn't seen.}

\item{print_info}{If TRUE, function will output information about the model as it is trained.}

\item{batch_size}{Size of the batch for stochastic gradient descent.}

\item{decay_rate}{A modification to the learning rate that decreases the learning rate as more and more learning
iterations are completed.}

\item{func_resp_method}{Set to 1 by default. In the future, this will be set to 2 for an alternative functional response
approach.}

\item{covariate_scaling}{If TRUE, then data will be internally scaled before model development.}

\item{raw_data}{If TRUE, then user does not need to create functional observations beforehand. The function will
internally take care of that pre-processing.}
}
\value{
The following are returned.

\code{predicted_folds} -- The predicted scalar values in each fold.

\code{true_folds} -- The true values of the response in each fold.

\code{MSPE} -- A list object containing the MSPE in each fold and the overall cross-validated MSPE.

\code{fold_indices} -- The generated indices for each fold; for replication purposes.
}
\description{
This is a convenience function for the user. The inputs are largely the same as the \code{\link[=fnn.fit]{fnn.fit()}} function with the
additional parameter of fold choice. This function only works for scalar responses.
}
\details{
No additional details for now.
}
\examples{
\donttest{
# Libraries
library(fda)

# Loading data
data("daily")

# Creating functional data
nbasis = 65
temp_data = array(dim = c(nbasis, 35, 1))
tempbasis65  = create.fourier.basis(c(0,365), nbasis)
tempbasis7 = create.bspline.basis(c(0,365), 7, norder = 4)
timepts = seq(1, 365, 1)
temp_fd = Data2fd(timepts, daily$tempav, tempbasis65)
prec_fd = Data2fd(timepts, daily$precav, tempbasis7)
prec_fd$coefs = scale(prec_fd$coefs)

# Data set up
temp_data[,,1] = temp_fd$coefs
resp_mat = prec_fd$coefs

# Non functional covariate
weather_scalar = data.frame(total_prec = apply(daily$precav, 2, sum))

# Setting up data to pass in to function
weather_data_full <- array(dim = c(nbasis, ncol(temp_data), 1))
weather_data_full[,,1] = temp_data
scalar_full = data.frame(weather_scalar[,1])
total_prec = apply(daily$precav, 2, mean)

# cross-validating
cv_example <- fnn.cv(nfolds = 5,
                     resp = total_prec,
                     func_cov = weather_data_full,
                     scalar_cov = scalar_full,
                     domain_range = list(c(1, 365)),
                     learn_rate = 0.001)
}

}
