#' Compute the Total Time-on-Test (TTOT) Statistic
#'
#' Calculates the Total Time-on-Test (TTOT) statistic for a given vector of event times and censoring indicators. The TTOT function is commonly used in survival analysis for visualizing and assessing the cumulative hazard or comparing survival patterns between groups.
#' @param data A data frame containing the input survival data.
#'   The first column should be named \code{"Time"} and contain the observed survival times
#'   (e.g., in months or years). The second column should be named \code{"Status"}, where
#'   0 indicates a censored observation and 1 indicates an event.
#' @param tA Numeric. The starting time point for computing the Total Time-on-Test (TTOT) function.
#' @param tB Numeric. The ending time point for computing the TTOT function.
#' @param t Numeric. The time point(s) at which the TTOT function is evaluated.
#' @return This function calculates the total time-on-test (TTOT) for a given dataset within the time interval (tA,tB).
#' @examples
#' # Generate sample survival data
#' set.seed(123)
#' sim_time <- generate_pe(n = 20, t = 1, lambda1 = 1, lambda2 = 0.5)
#' sim_status <- sample(c(0, 1), 20, replace = TRUE)
#' Data <- data.frame(Time = sim_time, Status = sim_status)
#' # Compute TTOT when time = 1
#' TTOT_fun(Data, t = 1)
#' @export
TTOT_fun <- function(data, tA=0, tB=10000, t) {
  # Ensure the input matrix has the correct columns
  if (!all(c("Time", "Status") %in% colnames(data))) {
    stop("Input data must have columns named 'Time' and 'Status'.")
  }

  # Extract Time and Event columns
  Time <- data$Time
  Status <- data$Status

  # Function to calculate TTOT for a given interval
  calc_TTOT <- function(Time, tA, tB) {
    sum(pmax(0, pmin(Time, tB) - tA))
  }

  # Calculate TTOT and event count for (tA, t]
  TTOT_1 <- calc_TTOT(Time, tA, t)
  event_count_1 <- sum(Time > tA & Time <= t & Status == 1)

  # Calculate TTOT and event count for (t, tB]
  TTOT_2 <- calc_TTOT(Time, t, tB)
  event_count_2 <- sum(Time > t & Time <= tB & Status == 1)

  # Return results as a list
  result <- list(
    TTOT_1 = TTOT_1,
    event_count_1 = event_count_1,
    TTOT_2 = TTOT_2,
    event_count_2 = event_count_2
  )

  return(result)
}
