% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/main_funcs.R
\name{CRT_pval}
\alias{CRT_pval}
\title{Testing whether factor X matters in Conjoint Experiments}
\usage{
CRT_pval(
  formula,
  data,
  X,
  left,
  right,
  design = "Uniform",
  p = NULL,
  constraint_randomization = NULL,
  supplyown_resamples = NULL,
  profileorder_constraint = TRUE,
  in_levs = NULL,
  forced_var = NULL,
  non_factor = NULL,
  B = 200,
  parallel = TRUE,
  num_cores = 2,
  nfolds = 3,
  lambda = c(20, 30, 40),
  tol = 0.001,
  speedup = TRUE,
  seed = sample(c(1:1000), size = 1),
  analysis = 0,
  verbose = TRUE
)
}
\arguments{
\item{formula}{A formula object specifying the outcome variable on the left-hand side
and factors of (X,Z) and respondent characteristics (V) in the right hand side.
RHS variables should be separated by + signs and should only contain either left
or right for each (X,Z).
For example Y ~ Country_left + Education_left is sufficient as opposed to
Y ~ Country_left + Country_right + Education_left + Education_right}

\item{data}{A dataframe containing outcome variable and all factors (X,Z,V)
(including both left and right profile factors). All (X,Z,V) listed in
the formula above are expected to be of class factor unless explicitly stated
in non_factor input.}

\item{X}{Character string specifying the variable of interest. This character
should match column name in data. For example "Country_left" is sufficient.}

\item{left}{Vector of column names of data that corresponds to the left profile factors}

\item{right}{Vector of column names of data that corresponds to the right profile factors.
NOTE: left and right are assumed to be the same length and the
order should correspond to the same variables. For example left = c("Country_left",
"Education_left") and right = c("Country_right", "Education_right")}

\item{design}{A character string of one of the following options: "Uniform",
"Constrained Uniform", "Nonuniform", "Manual". "Uniform" refers to a completely uniform
design where all (X,Z) are sampled uniformly. "Nonuniform" refers to a design where all
(X,Z) are sampled independently but the levels of X are not sampled uniformly.
If design="Nonuniform", then user should supply the non-uniform probability weights in p.
If in_levs is not NULL, then length of p should match in_levs. "Constrained Uniform"
refers to a dependent randomization design where some levels of X are only
possible based on certain levels of Z. If design="Constrained Uniform"
user should supply constraint_randomization list indicating the dependencies.
See examples below.
"Manual" refers to more complex conjoint designs, where the user will supply
their own resamples in supplyown_resamples input. Default is design="Uniform"}

\item{p}{A vector of nonuniform probability weights used when design="Nonuniform".
Length of p should match number of levels of X or length of in_levs.}

\item{constraint_randomization}{List containing levels of X that can only be
sampled with certain values of Z (used when design="Constrained Uniform").
The first element of constraint_randomization should contain the levels of X
that can only be sampled with certain values of Z, which are included in the
second element of the list. See example below.}

\item{supplyown_resamples}{List of length B that contains own resamples of X
when design="Manual". Each element of list should contain a dataframe
with the same number of rows of data and two columns for the left and right
profile values of X.}

\item{profileorder_constraint}{Boolean indicating whether to enforce profile
order constraint (default = TRUE)}

\item{in_levs}{A vector of strings which indicates a subset of the levels of X to
test for. See example below.}

\item{forced_var}{A character string indicating column name of Z or V that user
wishes to force an interaction with.}

\item{non_factor}{A vector of strings indicating columns of data that are not
factors. This should only be used for respondent characteristics (V) that are
not factors. For example non_factor = "Respondent_Age".}

\item{B}{Numeric integer value indicating the number of resamples for the CRT
procedure. Default value is B=200.}

\item{parallel}{Boolean indicating whether parallel computing should be used.
Default value is TRUE.}

\item{num_cores}{Numeric integer indicating number of cores to use when parallel=TRUE.
num_cores should not exceed the number of cores the user's machine can handle. Default is 2.}

\item{nfolds}{Numeric integer indicating number of cross-validation folds. Default is 3.}

\item{lambda}{Numeric vector indicating lambda used for cross-validation for
HierNet fit. Default lambda=c(20,30,40).}

\item{tol}{Numeric value indicating acceptable tolerance for terminating optimization
fit for HierNet. Default is tol=1e-3. WARNING: Do not increase as it greatly
increases computation time.}

\item{speedup}{Boolean indicating whether to employ computational tricks to
make function run faster. It is always recommended to use default speedup=TRUE.}

\item{seed}{Seed used for CRT procedure}

\item{analysis}{Numeric integer indicating whether to return the top x number
of strongest interactions that contributed to the the observed test statistic.
Default analysis = 0 to not return any top interactions. If analysis > 0,
for example analysis = 2, then the top two strongest interactions contribution to
the test statistic along with which interaction is returned.
NOTE: this is purely for exploratory analysis.}

\item{verbose}{Boolean indicating verbose output. Default verbose=TRUE}
}
\value{
A list containing: \item{p_val}{A numeric value for the p-value testing
Y independent of X given Z.}
\item{obs_test_stat}{A numeric value for the observed test statistic. If analysis
is > 0, obs_test_stat will contain a list detailing the contribution of the main effects
interaction effects and the top interactions.}
\item{resampled_test_stat}{Matrix containing all the B resampled test statistics}
\item{tol}{Tolerance used for HierNet}
\item{lam}{Best cross-validated lambda}
\item{hiernet_fit}{An object of class hiernet that contains the hiernet fit for
the observed test statistic}
\item{seed}{Seed used}
\item{elapsed_time}{Elapsed time}
}
\description{
This function takes a conjoint dataset and returns the p-value when using the
CRT to test if Y is independent of X given Z using HierNet test statistic.
The function requires user to specify the outcome, all factors used in the
conjoint experiment, and any additional respondent characteristics. By default,
this function assumes a uniform randomization of factor levels. In addition,
the function assumes the forced choice conjoint experiment and consequently assumes
the data to contain the left and right profile factors in separate columns in the
supplied dataset.
}
\examples{
# Subset of Immigration Choice Conjoint Experiment Data from Hainmueller et. al. (2014).
data("immigrationdata")
form = formula("Y ~ FeatEd + FeatGender + FeatCountry + FeatReason + FeatJob +
FeatExp + FeatPlans + FeatTrips + FeatLang + ppage + ppeducat + ppethm + ppgender")
left = colnames(immigrationdata)[1:9]
right = colnames(immigrationdata)[10:18]

\donttest{
# Testing whether edcuation matters for immigration preferences
education_test = CRT_pval(formula = form, data = immigrationdata, X = "FeatEd",
 left = left, right = right, non_factor = "ppage", B = 100, analysis = 2)
education_test$p_val
}

# Testing whether job matters for immigration preferences
constraint_randomization = list() # (Job has dependent randomization scheme)
constraint_randomization[["FeatJob"]] = c("Financial analyst","Computer programmer",
"Research scientist","Doctor")
constraint_randomization[["FeatEd"]] = c("Equivalent to completing two years of
college in the US", "Equivalent to completing a graduate degree in the US",
 "Equivalent to completing a college degree in the US")
\donttest{
job_test = CRT_pval(formula = form, data = immigrationdata, X = "FeatJob",
left = left, right = right, design = "Constrained Uniform",
constraint_randomization = constraint_randomization, non_factor = "ppage", B = 100)
job_test$p_val
}


# Testing whether Mexican and European immigrants are treated indistinguishably
country_data = immigrationdata
country_data$FeatCountry = as.character(country_data$FeatCountry)
country_data$FeatCountry_2 = as.character(country_data$FeatCountry_2)
country_data$FeatCountry[country_data$FeatCountry \%in\% c("Germany", "France",
"Poland")] = "Europe"
country_data$FeatCountry_2[country_data$FeatCountry_2 \%in\% c("Germany", "France",
 "Poland")] = "Europe"
country_data$FeatCountry = factor(country_data$FeatCountry)
country_data$FeatCountry_2 = factor(country_data$FeatCountry_2)
\donttest{
mexico_Europe_test = CRT_pval(formula = form, data = country_data, X = "FeatCountry",
left = left, right = right, design = "Nonuniform",
in_levs = c("Mexico", "Europe"), p = c(0.25, 0.75), non_factor = "ppage", B = 100,
analysis = 2)
}
\donttest{
# example case with supplying own resamples
resample_Mexico_Europe = function(country_data) {
 resamples_1 = sample(c("Mexico", "Europe"), size = nrow(country_data),
 replace = TRUE, p = c(0.25, 0.75))
 resamples_2 = sample(c("Mexico", "Europe"), size = nrow(country_data),
 replace = TRUE, p = c(0.25, 0.75))
 resample_df = data.frame(resamples_1, resamples_2)
 return(resample_df)
}
own_resamples = list()
for (i in 1:100) {
   own_resamples[[i]] = resample_Mexico_Europe(country_data)
}
mexico_Europe_test = CRT_pval(formula = form, data = country_data, X = "FeatCountry",
left = left, right = right, design = "Manual",
in_levs = c("Mexico", "Europe"), supplyown_resamples = own_resamples,
non_factor = "ppage", B = 100, analysis = 2)
}
# example case with forcing with candidate gender
\donttest{
mexico_Europe_test_force = CRT_pval(formula = form, data = country_data,
X = "FeatCountry", left = left, right = right, design = "Nonuniform",
in_levs = c("Mexico", "Europe"), p = c(0.25, 0.75), forced_var = "FeatGender",
non_factor = "ppage", B = 100)
}
}
\references{
Ham, D., Janson, L., and Imai, K. (2022)
Using Machine Learning to Test Causal Hypotheses in Conjoint Analysis
}
