# @rdname check_visits_within_span_MH
check_visits_within_span_MH <- function(dates, n_MH, t_MH) {
  dates <- sort(dates)  # Ensure dates are sorted
  for (i in seq_along(dates)) {
    if (sum(dates >= dates[i] & dates <= dates[i] + t_MH) >= n_MH) {
      return(TRUE)
    }
  }
  return(FALSE)
}

# @rdname compute_MH_status
compute_MH_status <- function(Diagnostic_H, Diagnostic_P, n_MHH, n_MHP, t_MH) {
  # Check matches within span for Diagnostic_H and Diagnostic_P
  match_H = check_visits_within_span_MH(Diagnostic_H, n_MHH, t_MH)
  match_P = check_visits_within_span_MH(Diagnostic_P, n_MHP, t_MH)

  # Determine status
  if (match_H || match_P) {
    return("YES")
  } else {
    return("NO")
  }
}


#' @rdname MH_status
#' @title Mental Health status detection in North American Healthcare Administrative Databases
#' @description Mental Health status is detected in North American Healthcare Administrative Databases  using clinician's parameters of interest such as minimum number of hospital visits, minimum number of physician services, time lag between them and plausible ICD diagnostics.
#' @param inputdata  a dataframe including columns: ClientID,  VisitDate,  Diagnostic_H, and Diagnostic_P
#' @param n_MHH minimum number of potential mental health related hospital visits
#' @param n_MHP minimum number of potential mental health related medical service physician visits
#' @param t_MH maximum time lag (in days) between all hospital visits and all medical service physician visits
#' @param ICD_MH plausible list of Mental Health status diagnostic codes
#' @returns a dataframe matrix with  Clients' ID, earliest date of Mental Health, latest date of Mental Health, and Mental Health status
#'
#' @references
#' Khan S. (2017). Concurrent mental and substance use disorders in Canada. Health reports, 28(8), 3–8, Ottawa, ON, Canada, PMID: 29044442. https://pubmed.ncbi.nlm.nih.gov/29044442/
#' Canadian Institute for Health Information. (2022). Canadian coding standards for version 2022 ICD-10-CA and CCI. Canadian Institute for Health Information. Ottawa, Ontario, Canada. https://www.cihi.ca/en/version-2022-icd-10-cacci-classifications-canadian-coding-standards-and-related-products
#' Centers for Disease Control and Prevention. (2024). International classification of diseases, tenth revision, clinical modification (ICD-10-CM) 2024. National Center for Health Statistics. Atlanta, GA, United States. https://www.cdc.gov/nchs/icd/icd-10-cm/index.html
#' Casillas, S. M., Scholl, L., Mustaquim, D., & Vivolo-Kantor, A. (2022). Analysis of trends and usage of ICD-10-CM discharge diagnosis codes for poisonings by fentanyl, tramadol, and other synthetic narcotics in emergency department data. Addictive Behaviors Reports, 16, 100464. Atlanta, GA, USA. https://doi.org/10.1016/j.abrep.2022.100464
#'
#' @import dplyr
#' @import tidyr
#' @import purrr
#'
#' @examples
#' data(SampleRWD)
#' myexample <- SampleRWD[,c(1:4)]
#' SampleMH_1 <- MH_status(myexample,
#' n_MHH=1, n_MHP=1, t_MH=60,
#' ICD_MH=c("F060","F063","F064", "F067"))
#' head(SampleMH_1)
#'
#' @export
#'

MH_status <- function(inputdata, n_MHH, n_MHP, t_MH, ICD_MH) {

  # Check matches and compute status(Check if the first 3/4/5 digits are in the ICD10ICD09_MH codes)
  result <- inputdata %>%
    mutate(
      Diagnostic_H_match = sapply(strsplit(.data$Diagnostic_H, ","), function(x) {
        any(sapply(3:5, function(n) any(substr(x, 1, n) %in% ICD_MH)))
      }),
      Diagnostic_P_match = sapply(strsplit(.data$Diagnostic_P, ","), function(x) {
        any(sapply(3:5, function(n) any(substr(x, 1, n) %in% ICD_MH)))
      })
    ) %>%
    group_by(.data$ClientID) %>%
    summarise(
      Diagnostic_H_filtered = list(.data$VisitDate[.data$Diagnostic_H_match]),
      Diagnostic_P_filtered = list(.data$VisitDate[.data$Diagnostic_P_match]),
      earliestdate_MH = ifelse(
        any(.data$Diagnostic_H_match | .data$Diagnostic_P_match),
        format(min(.data$VisitDate[.data$Diagnostic_H_match | .data$Diagnostic_P_match], na.rm = TRUE), "%Y-%m-%d"),
        NA
      ),
      latestdate_MH = ifelse(
        any(.data$Diagnostic_H_match | .data$Diagnostic_P_match),
        format(max(.data$VisitDate[.data$Diagnostic_H_match | .data$Diagnostic_P_match], na.rm = TRUE), "%Y-%m-%d"),
        NA
      ),
      MH_status = compute_MH_status(
        unlist(.data$Diagnostic_H_filtered),
        unlist(.data$Diagnostic_P_filtered),
        n_MHH,
        n_MHP,
        t_MH
      )
    ) %>%
    select(.data$ClientID, .data$earliestdate_MH, .data$latestdate_MH, .data$MH_status)

  return(result)
}

