#' @import pracma
require(pracma)
#' Simulate model realization
#'
#' @importFrom stats fft
#' @importFrom stats rexp
#' @importFrom stats rnorm
#' @importFrom stats runif
#' @importFrom stats rpois
#' @importFrom stats rgamma
#' @importFrom utils setTxtProgressBar
#' @importFrom utils txtProgressBar
#' @description Simulates a realization of the stochastic 3D structure model on a 3D grid. The default parameters are those fitted to 3D image data of particular cathode material in Gräfensteiner et al. 2024. For a detailed interpretation of model parameters, we refer to Section 3.1.5.
#' @param lambdaX intensity of the Boolean model mimicking the graphite  particles (default: 0.00000000006355 nm^-3).
#' @param alpha1 shape parameter of Gamma distribution modeling the length of one semi-axis of the ellipsoidal grain (default: 205 nm). 
#' @param alpha2 shape parameter of Gamma distribution modeling the length of one semi-axis of the ellipsoidal grain (default: 3944 nm). 
#' @param gamma scale parameter of Gamma distribution modeling the length of all semi-axes of the ellipsoidal grain (default: 1.971).
#' @param mu threshold of the excursion set model for the mixture of carbon black and binder (default: 0.499).
#' @param eta parameter of the covariance function of the underlying Gaussian random field. This parameter controls how fine the mixture of carbon black and binder is (default: 0.0127 nm^-1).
#' @param lambdaY intensity of the Boolean model mimicking the large pore regions (default: 0.000000009340 nm^-3).
#' @param theta rate parameter of the Exponential distribution modeling the radii of large pores (default: 0.01052632 nm^-1).
#' @param size the size of the cubic grid, on which the model is simulated (default: 800).
#' @param edge the amount of edge correction used for model simulation (default: 150). The model is realized on a cubic grid, which is extended in each direction by a ceretain number of additional grid points, given by the parameter edge. This avoids edge effects at the boundary.
#' @param resolution the physical distance between to grid points in nm (default: 20). Be careful with changing resolutions, see Figure 5 and the  corresponding discussion in Gräfensteiner et al. 2024.
#' @param greyscale this parameter is a 2D vector, which contains the labels for the mixture of carbon black and binder (first entry) and graphite particles (second entry). Note that the pores are always labeled with zero. (default: (255, 255))
#' @param progress Binary parameter. If true, information regarding the progress of the current model realization is printed. (default: FALSE)
#' @return Returns a model realization in form of a cubic 3D matrix. The length in each direction is given by the parameter 'size'. At each gridpoint the greyscale value of the corresponding phase is stored. The value 0 indicates pores, while the parameters 'greyscale[1]' and 'greyscale[2]' indicate the carbon black/binder-mixture and the graphite particles, respectively.
#' @references P. Gräfensteiner, M. Osenberg, A. Hilger, N. Bohn, J. R. Binder, I. Manke, V. Schmidt, M. Neumann (2024) <doi:10.48550/arXiv.2409.11080>
#' @examples Xi <- realize(size = 10, edge = 10)
#' @export
realize <- function(lambdaX = 0.00000000006355, alpha1 = 205, alpha2 = 3944, gamma = 1.971, mu = 0.499, eta = 0.0127, lambdaY = 0.000000009340, theta = 0.01052632, size = 800, edge = 150, resolution = 20, greyscale = c(255, 255), progress = FALSE){
  ##############################################################################
  # 0. Initialize image size and rescale parameters
  ##############################################################################
  N <- size 
  lambdaX <- lambdaX * resolution^3
  alpha1 <- alpha1 / resolution
  alpha2 <- alpha2 / resolution
  eta <- eta * resolution
  lambdaY <- lambdaY * resolution^3
  theta <- theta * resolution
  
  ##############################################################################
  # 1. Simulate Gaussian random field
  ##############################################################################
  # Create 3D grid for the covariance function
  if(progress){
    message("\n Simulate Gaussian excursion set \n")
    pb<-txtProgressBar(min=0,max=8,style=3)
  }
  
  
  
  
  iota <-0 
  kx <- array(rep(seq(-N/2, N/2 - 1, length.out = N), each = N^2), dim = c(N, N, N))
  iota <- 1
  if(progress){
    setTxtProgressBar(pb, iota)
  }
  
  ky <- array(rep(rep(seq(-N/2, N/2 - 1, length.out = N), each = N), times = N), dim = c(N, N, N))
  iota <- 2
  if(progress){
    setTxtProgressBar(pb, iota)
  }
  kz <- array(rep(seq(-N/2, N/2 - 1, length.out = N), times = N^2), dim = c(N, N, N))
  iota <- 3
  if(progress){
    setTxtProgressBar(pb, iota)
  }
  h <- sqrt(kx^2 + ky^2 + kz^2)
  
  # Define the covariance function and compute the power spectrum
  C <- 1 / (1 + (eta * h)^2)
  power_spectrum <- fft(C, inverse = TRUE)
  iota <- 4
  if(progress){
    setTxtProgressBar(pb, iota)
  }
  
  # Generate the random field in Fourier space (complex Gaussian random field)
  random_field_fft <- sqrt(power_spectrum) * (rnorm(N^3) + 1i * rnorm(N^3))
  iota <- 5
  if(progress){
    setTxtProgressBar(pb, iota)
  }
  random_field_fft <- array(random_field_fft, dim = c(N, N, N))
  iota <- 6
  if(progress){
    setTxtProgressBar(pb, iota)
  }
  
  # Apply the inverse Fourier transform to get the 3D Gaussian random field
  Z <- Re(fft(random_field_fft, inverse = TRUE) / N^1.5)
  iota <- 7
  if(progress){
    setTxtProgressBar(pb, iota)
  }
  Xi <- greyscale[1] * (Z >= mu)
  iota <- 8
  if(progress){
    setTxtProgressBar(pb, iota)
  }
  
  ##############################################################################
  # 2. Generate large pore regions 
  ##############################################################################
  # Simulate Poisson process
  M <- rpois(n = 1, lambda = lambdaY * (N + 2 * edge)^3)
  
  if(progress){
    message("\n Simulate large pores \n")
  }
  
  Y <- matrix(runif(M*3, min = -1 * edge, max = N + edge), nrow = M, ncol = 3)
  # Simulate random radii
  R <- rexp(n = M, rate <- theta)
  
  
  # Discretize the realization of the Boolean model
  if(M > 0){
    if(progress){
      pb<-txtProgressBar(min=0,max=M,style=3)
    }
    
    for (m in 1:M) {
      xlow = max(0, floor(Y[m, 1] - R[m]))
      xhigh = min(N, ceil(Y[m, 1] + R[m]))
      ylow = max(0, floor(Y[m, 2] - R[m]))
      yhigh = min(N, ceil(Y[m, 2] + R[m]))
      zlow = max(0, floor(Y[m, 3] - R[m]))
      zhigh = min(N, ceil(Y[m, 3] + R[m]))
      for (i1 in xlow:xhigh) {
        for (i2 in ylow:yhigh) {
          for (i3 in zlow:zhigh) {
            if ((i1 - Y[m, 1]) ^ 2 + (i2 - Y[m, 2]) ^ 2 + (i3 - Y[m, 3]) ^ 2 <= R[m] ^ 2) {
              Xi[i1, i2, i3] = 0
            }
          }
        }
      }
      if(progress){
        setTxtProgressBar(pb, m)
      }
    }
  }
  ##############################################################################
  # 3. Generate graphite particles
  ##############################################################################
  # Simulate Poisson process
  if(progress){
    message("\n Simulate graphite particles \n")
  }
  L <- rpois(n = 1, lambda = lambdaX * (N + 2 * edge) ^ 3)
  X <- matrix(runif(L * 3, min = -1 * edge, max = N + edge), nrow = L, ncol = 3)
  # Simulate random length of half-axes
  A <- rgamma(n = L, shape = alpha1, rate = gamma)
  C <- rgamma(n = L, shape = alpha2, rate = gamma) 
  # Discretize the realization of the Boolean model
  if(L > 0){
    if(progress){
      pb<-txtProgressBar(min=0,max=L,style=3)
    }
    for (l in 1:L) {
      
      a <- max(A[l], C[l])
      c <- min(A[l], C[l])
      
      Diag <- diag(c(1 / (a ^ 2), 1 / (a ^ 2), 1 / (c ^ 2)))
      Diag_inv <- diag(c(a^2, a^2, c^2))
      
      Rot <- randortho(n = 3, type = "orthonormal")
      if(det(Rot) == -1){
        Rot = -1 * Rot
      }
      Q = t(Rot) %*% Diag %*% Rot
      Qinv = t(Rot) %*% Diag_inv %*% Rot
      
      box_length1 <- sqrt(Qinv[1, 1])
      box_length2 <- sqrt(Qinv[2, 2])
      box_length3 <- sqrt(Qinv[3, 3])
      
      xlow = max(0, floor(X[l, 1] - box_length1))
      xhigh = min(N, ceil(X[l, 1] + box_length1))
      ylow = max(0, floor(X[l, 2] - box_length2))
      yhigh = min(N, ceil(X[l, 2] + box_length2))
      zlow = max(0, floor(X[l, 3] - box_length3))
      zhigh = min(N, ceil(X[l, 3] + box_length3))
      
      midpoint <- matrix(c(X[l,1], X[l,2], X[l,3]), nrow = 3, ncol = 1)
      for (i1 in xlow:xhigh) {
        for (i2 in ylow:yhigh) {
          for (i3 in zlow:zhigh) {
            vec <- matrix(c(i1, i2, i3), nrow = 3, ncol = 1)
            if (t(vec - midpoint) %*% Q %*% (vec - midpoint) <= 1) {
              Xi[i1, i2, i3] = greyscale[2]
            }
          }
        }
      }
      if(progress){
        setTxtProgressBar(pb, l)
      }
    }
  }
  
  
  return(Xi)
}


