#' A function to fit the stochastic mortality model MLiLee.
#'
#' Carry out Bayesian estimation of the stochastic mortality \bold{model MLiLee} (Li and Lee, 2005). Note that if the number of strata is one, results from this model are essentially the same as the Lee-Carter model, fit_LC().
#' 
#' The model can be described mathematically as follows:
#' If \code{family="log"}, then
#' \deqn{d_{x,t,p} \sim \text{Poisson}(E^c_{x,t,p} m_{x,t,p}) , }
#' \deqn{\log(m_{x,t,p})=a_{x,p}+b_{x,p}k_{t,p}+B_xK_t , }
#' where \eqn{d_{x,t,p}} represents the number of deaths at age \eqn{x} in year \eqn{t} of stratum \eqn{p},
#' while \eqn{E^c_{x,t,p}} and \eqn{m_{x,t,p}} represents respectively the corresponding central exposed to risk and central mortality rate at age \eqn{x} in year \eqn{t} of stratum \eqn{p}.
#' Similarly, if \code{family="nb"}, then a negative binomial distribution is fitted, i.e.
#' \deqn{d_{x,t,p} \sim \text{Negative-Binomial}(\phi,\frac{\phi}{\phi+E^c_{x,t,p} m_{x,t,p}}) , }
#' \deqn{\log(m_{x,t,p})=a_{x,p}+b_{x,p}k_{t,p}+B_xK_t , }
#' where \eqn{\phi} is the overdispersion parameter. See Wong et al. (2018).
#' But if \code{family="binomial"}, then 
#' \deqn{d_{x,t,p} \sim \text{Binomial}(E^0_{x,t,p} , q_{x,t,p}) , }
#' \deqn{\text{logit}(q_{x,t,p})=a_{x,p}+b_{x,p}k_{t,p}+B_xK_t , }
#' where \eqn{q_{x,t,p}} represents the initial mortality rate at age \eqn{x} in year \eqn{t} of stratum \eqn{p},
#' while \eqn{E^0_{x,t,p}\approx E^c_{x,t,p}+\frac{1}{2}d_{x,t,p}} is the corresponding initial exposed to risk.
#' Constraints used are:
#' \deqn{\sum_{x} b_{x} = 1, \sum_{t,p} k_{t,p} = 0 .}
#' If \code{share_alpha=TRUE}, then the additive age-specific parameter is the same across all strata \eqn{p}, i. e.  
#' \deqn{a_{x}+b_{x,p}k_{t,p}+B_xK_t .}
#' If \code{forecast=TRUE}, then a time series model (an AR(1) with linear drift) will be fitted on both \eqn{k_{t,p}} and \eqn{K_t} as follows:
#' \deqn{k_{t,p} = \eta^k_1+\eta^k_2 t +\rho_k (k_{t-1,p}-(\eta^k_1+\eta^k_2 (t-1))) + \epsilon^k_{t,p} \text{ for }p=1,\ldots,P-1 \text{ and } t=1,\ldots,T,}
#' and
#' \deqn{K_{t} = \eta^K_1+\eta^K_2 t +\rho^K (K_{t-1}-(\eta^K_1+\eta^K_2 (t-1))) + \epsilon^K_{t} \text{ for }t=1,\ldots,T,}
#' where \eqn{\epsilon^k_{t,p}\sim N(0,\sigma_k^2)}, \eqn{\epsilon^K_{t}\sim N(0,\sigma_K^2)}, while \eqn{\eta^k_1,\eta^k_2,\rho_k,\sigma_k^2, \eta^K_1,\eta^K_2,\rho_K,\sigma_K^2} are additional parameters to be estimated.
#' In principle, there are many other options for forecasting the mortality time trend. But currently, we assume that this serves as a general purpose forecasting model for simplicity.
#' 
#' @references Li N., & Lee R. (2005). Coherent mortality forecasts for a group of populations: an extension of the Lee-Carter method. Demography. 42(3):575-94. \doi{https://doi.org/10.1353/dem.2005.0021}
#' @references Jackie S. T. Wong, Jonathan J. Forster, and Peter W. F. Smith. (2018). Bayesian mortality forecasting with overdispersion, Insurance: Mathematics and Economics, Volume 2018, Issue 3, 206-221. \doi{https://doi.org/10.1016/j.insmatheco.2017.09.023}
#' 
#' @param death death data that has been formatted through the function \code{preparedata_fn}.
#' @param expo expo data that has been formatted through the function \code{preparedata_fn}.
#' @param n_iter number of iterations to run. Default is \code{n_iter=10000}. 
#' @param family a string of characters that defines the family function associated with the mortality model. "poisson" would assume that deaths follow a Poisson distribution and use a log link; "binomial" would assume that deaths follow a Binomial distribution and a logit link; "nb" (default) would assume that deaths follow a Negative-Binomial distribution and a log link.
#' @param share_alpha a logical value indicating if \eqn{a_{x,p}} should be shared across all strata (see details below). Default is \code{FALSE}.
#' @param n.chain number of parallel chains for the model.
#' @param thin thinning interval for monitoring purposes.
#' @param n.adapt the number of iterations for adaptation. See \code{?rjags::adapt} for details.
#' @param forecast a logical value indicating if forecast is to be performed (default is \code{FALSE}). See below for details.
#' @param h a numeric value giving the number of years to forecast. Default is \code{h=5}.
#' @param quiet if TRUE then messages generated during compilation will be suppressed, as well as the progress bar during adaptation.
#' @return A list with components:
#' \describe{
#'   \item{\code{post_sample}}{An \code{mcmc.list} object containing the posterior samples generated.}
#'   \item{\code{param}}{A vector of character strings describing the names of model parameters.}
#'   \item{\code{death}}{The death data that was used.}
#'   \item{\code{expo}}{The expo data that was used.}
#'   \item{\code{family}}{The family function used.}
#'   \item{\code{forecast}}{A logical value indicating if forecast has been performed.}
#'   \item{\code{h}}{The forecast horizon used.}
#' }
#' @keywords bayesian estimation models
#' @concept stochastic mortality models
#' @concept parameter estimation
#' @concept Li-Lee
#' @importFrom stats dnbinom dbinom dpois quantile sd
#' @export
#' @examples
#' #load and prepare mortality data
#' data("dxt_array_product");data("Ext_array_product")
#' death<-preparedata_fn(dxt_array_product,strat_name = c("ACI","DB","SCI"),ages=35:65)
#' expo<-preparedata_fn(Ext_array_product,strat_name = c("ACI","DB","SCI"),ages=35:65)
#' 
#' #fit the model (negative-binomial family)
#' #NOTE: This is a toy example, please run it longer in practice.
#' fit_MLiLee_result<-fit_MLiLee(death=death,expo=expo,n_iter=50,n.adapt=50)
#' head(fit_MLiLee_result)
#' 
#' \donttest{
#' #if sharing the alphas (poisson family)
#' fit_MLiLee_result2<-fit_MLiLee(death=death,expo=expo,n_iter=1000,family="poisson",share_alpha=TRUE)
#' head(fit_MLiLee_result2)
#' 
#' #if forecast (poisson family)
#' fit_MLiLee_result3<-fit_MLiLee(death=death,expo=expo,n_iter=1000,family="poisson",forecast=TRUE)
#' plot_rates_fn(fit_MLiLee_result3)
#' plot_param_fn(fit_MLiLee_result3)
#' }

fit_MLiLee<-function(death,expo,n_iter=10000,family="nb",share_alpha=FALSE,n.chain=1,thin=1,n.adapt=1000,forecast=FALSE,h=5,quiet=FALSE){
  
  p<-death$n_strat
  A<-death$n_ages
  T<-death$n_years
  
  prior_mean_Beta=prior_mean_beta=rep(1/A,A-1)
  sigma2_beta<-0.001
  prior_prec_Beta=prior_prec_beta=solve(sigma2_beta*(diag(rep(1,A-1))-1/A*(matrix(1,nrow=A-1,ncol=A-1))))
  
  t<-(1:T)-mean(1:T)
  matrix_kappa_X<-matrix(c(rep(1,T),t),byrow=F,ncol=2)
  prior_prec_eta<-solve(matrix(c(400,0,0,2),nrow=2));prior_mean_eta<-c(0,0)
  
  if (!share_alpha){
    if (!forecast){
      
      if (family=="binomial"){
        expo_initial<-round(expo$data+0.5*death$data)
        data<-list(dxt=death$data,ext=expo_initial,A=A,T=T,p=p,prior_mean_Beta=prior_mean_Beta,prior_prec_Beta=prior_prec_Beta,prior_mean_beta=prior_mean_beta,prior_prec_beta=prior_prec_beta,matrix_kappa_X=matrix_kappa_X,prior_mean_eta=prior_mean_eta,prior_prec_eta=prior_prec_eta)
        inits<-function() (list(alpha=matrix(0,nrow=p,ncol=A),Beta_rest=rep(1/A,A-1),beta_rest_mat=matrix(1/A,nrow=p-1,ncol=A-1),Kappa_rest=rep(0,T-1),kappa_rest_mat=matrix(0,nrow=p-1,ncol=T-1),rho_kappa=0.5,eta_kappa=c(0,0),i_sigma2_kappa=0.1,rho_Kappa=0.5,eta_Kappa=c(0,0),i_sigma2_Kappa=0.1))
        if (p==1){  
          vars<-c("q","alpha","Beta","Kappa","eta_Kappa","rho_Kappa","sigma2_Kappa")
        } else {
          vars<-c("q","alpha","beta","kappa","Beta","Kappa","eta_kappa","rho_kappa","sigma2_kappa","eta_Kappa","rho_Kappa","sigma2_Kappa")
        }
        logit_LC_MLiLee_jags<-rjags::jags.model(system.file("models/logit_LC_MLiLee.jags", package = "BayesMoFo"),data=data,inits=inits,n.chain=n.chain,n.adapt=n.adapt,quiet=quiet)
        result_jags<-rjags::coda.samples(logit_LC_MLiLee_jags,vars,n.iter=n_iter,thin=thin)
      }
      if (family=="poisson"){
        data<-list(dxt=death$data,ext=expo$data,A=A,T=T,p=p,prior_mean_Beta=prior_mean_Beta,prior_prec_Beta=prior_prec_Beta,prior_mean_beta=prior_mean_beta,prior_prec_beta=prior_prec_beta,matrix_kappa_X=matrix_kappa_X,prior_mean_eta=prior_mean_eta,prior_prec_eta=prior_prec_eta)
        inits<-function() (list(alpha=matrix(0,nrow=p,ncol=A),Beta_rest=rep(1/A,A-1),beta_rest_mat=matrix(1/A,nrow=p-1,ncol=A-1),Kappa_rest=rep(0,T-1),kappa_rest_mat=matrix(0,nrow=p-1,ncol=T-1),rho_kappa=0.5,eta_kappa=c(0,0),i_sigma2_kappa=0.1,rho_Kappa=0.5,eta_Kappa=c(0,0),i_sigma2_Kappa=0.1))
        if (p==1){
          vars<-c("q","alpha","Beta","Kappa","eta_Kappa","rho_Kappa","sigma2_Kappa")
        } else {
          vars<-c("q","alpha","beta","kappa","Beta","Kappa","eta_kappa","rho_kappa","sigma2_kappa","eta_Kappa","rho_Kappa","sigma2_Kappa")
        }
        log_LC_MLiLee_jags<-rjags::jags.model(system.file("models/log_LC_MLiLee.jags", package = "BayesMoFo"),data=data,inits=inits,n.chain=n.chain,n.adapt=n.adapt,quiet=quiet)
        result_jags<-rjags::coda.samples(log_LC_MLiLee_jags,vars,n.iter=n_iter,thin=thin)
      }
      if (family=="nb"){
        data<-list(dxt=death$data,ext=expo$data,A=A,T=T,p=p,prior_mean_Beta=prior_mean_Beta,prior_prec_Beta=prior_prec_Beta,prior_mean_beta=prior_mean_beta,prior_prec_beta=prior_prec_beta,matrix_kappa_X=matrix_kappa_X,prior_mean_eta=prior_mean_eta,prior_prec_eta=prior_prec_eta)
        inits<-function() (list(alpha=matrix(0,nrow=p,ncol=A),Beta_rest=rep(1/A,A-1),beta_rest_mat=matrix(1/A,nrow=p-1,ncol=A-1),Kappa_rest=rep(0,T-1),kappa_rest_mat=matrix(0,nrow=p-1,ncol=T-1),rho_kappa=0.5,eta_kappa=c(0,0),i_sigma2_kappa=0.1,rho_Kappa=0.5,eta_Kappa=c(0,0),i_sigma2_Kappa=0.1,phi=100))
        if (p==1){
          vars<-c("q","alpha","Beta","Kappa","eta_Kappa","rho_Kappa","sigma2_Kappa","phi")
        } else {
          vars<-c("q","alpha","beta","kappa","Beta","Kappa","eta_kappa","rho_kappa","sigma2_kappa","eta_Kappa","rho_Kappa","sigma2_Kappa","phi") 
        }
        nb_LC_MLiLee_jags<-rjags::jags.model(system.file("models/nb_LC_MLiLee.jags", package = "BayesMoFo"),data=data,inits=inits,n.chain=n.chain,n.adapt=n.adapt,quiet=quiet)
        result_jags<-rjags::coda.samples(nb_LC_MLiLee_jags,vars,n.iter=n_iter,thin=thin)
      }
    } else {
      
      #forecast
      t<-(1:T)-mean(1:T)
      matrix_kappa_X<-matrix(c(rep(1,T+h),c(t,t[T]+1:h)),byrow=F,ncol=2)
      prior_prec_eta<-solve(matrix(c(400,0,0,2),nrow=2));prior_mean_eta<-c(0,0)
      
      death_forecast<-array(dim=c(p,A,T+h));expo_forecast<-array(dim=c(p,A,T+h))
      death_forecast[,,1:T]<-death$data
      death_forecast[,,(T+1):(T+h)]<-NA
      expo_forecast[,,1:T]<-expo$data
      expo_forecast[,,(T+1):(T+h)]<-expo$data[,,T]
      
      if (family=="binomial"){
        expo_forecast_initial<-expo_forecast
        expo_forecast_initial[,,1:T]<-round(expo_forecast[,,1:T,drop=FALSE]+0.5*death$data)
        expo_forecast_initial[,,(T+1):(T+h)]<-round(expo$data[,,T]+0.5*death$data[,,T])
        
        data<-list(dxt=death_forecast,ext=expo_forecast_initial,A=A,T=T,p=p,h=h,prior_mean_Beta=prior_mean_Beta,prior_prec_Beta=prior_prec_Beta,prior_mean_beta=prior_mean_beta,prior_prec_beta=prior_prec_beta,matrix_kappa_X=matrix_kappa_X,prior_mean_eta=prior_mean_eta,prior_prec_eta=prior_prec_eta)
        inits<-function() (list(alpha=matrix(0,nrow=p,ncol=A),Beta_rest=rep(1/A,A-1),beta_rest_mat=matrix(1/A,nrow=p-1,ncol=(A-1)),Kappa_rest=rep(0,T-1),kappa_rest_mat=matrix(0,nrow=p-1,ncol=(T-1)),rho_kappa=0.5,eta_kappa=c(0,0),i_sigma2_kappa=0.1,rho_Kappa=0.5,eta_Kappa=c(0,0),i_sigma2_Kappa=0.1))
        if (p==1){
          vars<-c("q","alpha","Beta","Kappa","eta_Kappa","rho_Kappa","sigma2_Kappa")
        } else {
          vars<-c("q","alpha","beta","kappa","Beta","Kappa","eta_kappa","rho_kappa","sigma2_kappa","eta_Kappa","rho_Kappa","sigma2_Kappa")
        }
        logit_LC_MLiLee_alt_forecast_jags<-rjags::jags.model(system.file("models/logit_LC_MLiLee_alt_forecast.jags",package="BayesMoFo"),data=data,inits=inits,n.chain=n.chain,n.adapt = n.adapt,quiet=quiet)
        result_jags<-rjags::coda.samples(logit_LC_MLiLee_alt_forecast_jags,vars,n.iter=n_iter,thin=thin)
        
      }
      if (family=="poisson"){
        data<-list(dxt=death_forecast,ext=expo_forecast,A=A,T=T,p=p,h=h,prior_mean_Beta=prior_mean_Beta,prior_prec_Beta=prior_prec_Beta,prior_mean_beta=prior_mean_beta,prior_prec_beta=prior_prec_beta,matrix_kappa_X=matrix_kappa_X,prior_mean_eta=prior_mean_eta,prior_prec_eta=prior_prec_eta)
        inits<-function() (list(alpha=matrix(0,nrow=p,ncol=A),Beta_rest=rep(1/A,A-1),beta_rest_mat=matrix(1/A,nrow=p-1,ncol=(A-1)),Kappa_rest=rep(0,T-1),kappa_rest_mat=matrix(0,nrow=p-1,ncol=(T-1)),rho_kappa=0.5,eta_kappa=c(0,0),i_sigma2_kappa=0.1,rho_Kappa=0.5,eta_Kappa=c(0,0),i_sigma2_Kappa=0.1))
        if (p==1){
          vars<-c("q","alpha","Beta","Kappa","eta_Kappa","rho_Kappa","sigma2_Kappa")
        } else {
          vars<-c("q","alpha","beta","kappa","Beta","Kappa","eta_kappa","rho_kappa","sigma2_kappa","eta_Kappa","rho_Kappa","sigma2_Kappa")
        }
        log_LC_MLiLee_alt_forecast_jags<-rjags::jags.model(system.file("models/log_LC_MLiLee_alt_forecast.jags",package="BayesMoFo"),data=data,inits=inits,n.chain=n.chain,n.adapt = n.adapt,quiet=quiet)
        result_jags<-rjags::coda.samples(log_LC_MLiLee_alt_forecast_jags,vars,n.iter=n_iter,thin=thin)
      }
      if (family=="nb"){
        data<-list(dxt=death_forecast,ext=expo_forecast,A=A,T=T,p=p,h=h,prior_mean_Beta=prior_mean_Beta,prior_prec_Beta=prior_prec_Beta,prior_mean_beta=prior_mean_beta,prior_prec_beta=prior_prec_beta,matrix_kappa_X=matrix_kappa_X,prior_mean_eta=prior_mean_eta,prior_prec_eta=prior_prec_eta)
        inits<-function() (list(alpha=matrix(0,nrow=p,ncol=A),Beta_rest=rep(1/A,A-1),beta_rest_mat=matrix(1/A,nrow=p-1,ncol=(A-1)),Kappa_rest=rep(0,T-1),kappa_rest_mat=matrix(0,nrow=p-1,ncol=(T-1)),rho_kappa=0.5,eta_kappa=c(0,0),i_sigma2_kappa=0.1,rho_Kappa=0.5,eta_Kappa=c(0,0),i_sigma2_Kappa=0.1,phi=100))
        if (p==1){
          vars<-c("q","alpha","Beta","Kappa","eta_Kappa","rho_Kappa","sigma2_Kappa","phi")
        } else {
          vars<-c("q","alpha","beta","kappa","Beta","Kappa","eta_kappa","rho_kappa","sigma2_kappa","eta_Kappa","rho_Kappa","sigma2_Kappa","phi")
        }
        nb_LC_MLiLee_alt_forecast_jags<-rjags::jags.model(system.file("models/nb_LC_MLiLee_alt_forecast.jags",package="BayesMoFo"),data=data,inits=inits,n.chain=n.chain,n.adapt = n.adapt,quiet=quiet)
        result_jags<-rjags::coda.samples(nb_LC_MLiLee_alt_forecast_jags,vars,n.iter=n_iter,thin=thin)
      }
    }
  } else {
    if (!forecast){
      
      if (family=="binomial"){
        expo_initial<-round(expo$data+0.5*death$data)
        data<-list(dxt=death$data,ext=expo_initial,A=A,T=T,p=p,prior_mean_Beta=prior_mean_Beta,prior_prec_Beta=prior_prec_Beta,prior_mean_beta=prior_mean_beta,prior_prec_beta=prior_prec_beta,matrix_kappa_X=matrix_kappa_X,prior_mean_eta=prior_mean_eta,prior_prec_eta=prior_prec_eta)
        inits_modified<-function() (list(alpha=rep(0,A),Beta_rest=rep(1/A,A-1),beta_rest_mat=matrix(1/A,nrow=p-1,ncol=A-1),Kappa_rest=rep(0,T-1),kappa_rest_mat=matrix(0,nrow=p-1,ncol=T-1),rho_kappa=0.5,eta_kappa=c(0,0),i_sigma2_kappa=0.1,rho_Kappa=0.5,eta_Kappa=c(0,0),i_sigma2_Kappa=0.1))
        if (p==1){  
          vars<-c("q","alpha","Beta","Kappa","eta_Kappa","rho_Kappa","sigma2_Kappa")
        } else {
          vars<-c("q","alpha","beta","kappa","Beta","Kappa","eta_kappa","rho_kappa","sigma2_kappa","eta_Kappa","rho_Kappa","sigma2_Kappa")
        }
        logit_LC_MLiLee_jags<-rjags::jags.model(system.file("models/logit_LC_MLiLee_modified.jags", package = "BayesMoFo"),data=data,inits=inits_modified,n.chain=n.chain,n.adapt=n.adapt,quiet=quiet)
        result_jags<-rjags::coda.samples(logit_LC_MLiLee_jags,vars,n.iter=n_iter,thin=thin)
      }
      if (family=="poisson"){
        data<-list(dxt=death$data,ext=expo$data,A=A,T=T,p=p,prior_mean_Beta=prior_mean_Beta,prior_prec_Beta=prior_prec_Beta,prior_mean_beta=prior_mean_beta,prior_prec_beta=prior_prec_beta,matrix_kappa_X=matrix_kappa_X,prior_mean_eta=prior_mean_eta,prior_prec_eta=prior_prec_eta)
        inits_modified<-function() (list(alpha=rep(0,A),Beta_rest=rep(1/A,A-1),beta_rest_mat=matrix(1/A,nrow=p-1,ncol=A-1),Kappa_rest=rep(0,T-1),kappa_rest_mat=matrix(0,nrow=p-1,ncol=T-1),rho_kappa=0.5,eta_kappa=c(0,0),i_sigma2_kappa=0.1,rho_Kappa=0.5,eta_Kappa=c(0,0),i_sigma2_Kappa=0.1))
        if (p==1){  
          vars<-c("q","alpha","Beta","Kappa","eta_Kappa","rho_Kappa","sigma2_Kappa")
        } else {
          vars<-c("q","alpha","beta","kappa","Beta","Kappa","eta_kappa","rho_kappa","sigma2_kappa","eta_Kappa","rho_Kappa","sigma2_Kappa")
        }
        log_LC_MLiLee_jags<-rjags::jags.model(system.file("models/log_LC_MLiLee_modified.jags", package = "BayesMoFo"),data=data,inits=inits_modified,n.chain=n.chain,n.adapt=n.adapt,quiet=quiet)
        result_jags<-rjags::coda.samples(log_LC_MLiLee_jags,vars,n.iter=n_iter,thin=thin)
      }
      if (family=="nb"){
        data<-list(dxt=death$data,ext=expo$data,A=A,T=T,p=p,prior_mean_Beta=prior_mean_Beta,prior_prec_Beta=prior_prec_Beta,prior_mean_beta=prior_mean_beta,prior_prec_beta=prior_prec_beta,matrix_kappa_X=matrix_kappa_X,prior_mean_eta=prior_mean_eta,prior_prec_eta=prior_prec_eta)
        inits_modified<-function() (list(alpha=rep(0,A),Beta_rest=rep(1/A,A-1),beta_rest_mat=matrix(1/A,nrow=p-1,ncol=A-1),Kappa_rest=rep(0,T-1),kappa_rest_mat=matrix(0,nrow=p-1,ncol=T-1),rho_kappa=0.5,eta_kappa=c(0,0),i_sigma2_kappa=0.1,rho_Kappa=0.5,eta_Kappa=c(0,0),i_sigma2_Kappa=0.1,phi=100))
        if (p==1){
          vars<-c("q","alpha","Beta","Kappa","eta_Kappa","rho_Kappa","sigma2_Kappa","phi")
        } else {
          vars<-c("q","alpha","beta","kappa","Beta","Kappa","eta_kappa","rho_kappa","sigma2_kappa","eta_Kappa","rho_Kappa","sigma2_Kappa","phi")
        }
        nb_LC_MLiLee_jags<-rjags::jags.model(system.file("models/nb_LC_MLiLee_modified.jags", package = "BayesMoFo"),data=data,inits=inits_modified,n.chain=n.chain,n.adapt=n.adapt,quiet=quiet)
        result_jags<-rjags::coda.samples(nb_LC_MLiLee_jags,vars,n.iter=n_iter,thin=thin)
      }
      
    } else {
      
      #forecast
      t<-(1:T)-mean(1:T)
      matrix_kappa_X<-matrix(c(rep(1,T+h),c(t,t[T]+1:h)),byrow=F,ncol=2)
      prior_prec_eta<-solve(matrix(c(400,0,0,2),nrow=2));prior_mean_eta<-c(0,0)
      
      death_forecast<-array(dim=c(p,A,T+h));expo_forecast<-array(dim=c(p,A,T+h))
      death_forecast[,,1:T]<-death$data
      death_forecast[,,(T+1):(T+h)]<-NA
      expo_forecast[,,1:T]<-expo$data
      expo_forecast[,,(T+1):(T+h)]<-expo$data[,,T]
      
      if (family=="binomial"){
        expo_forecast_initial<-expo_forecast
        expo_forecast_initial[,,1:T]<-round(expo_forecast[,,1:T,drop=FALSE]+0.5*death$data)
        expo_forecast_initial[,,(T+1):(T+h)]<-round(expo$data[,,T]+0.5*death$data[,,T])
        
        data<-list(dxt=death_forecast,ext=expo_forecast_initial,A=A,T=T,p=p,h=h,prior_mean_Beta=prior_mean_Beta,prior_prec_Beta=prior_prec_Beta,prior_mean_beta=prior_mean_beta,prior_prec_beta=prior_prec_beta,matrix_kappa_X=matrix_kappa_X,prior_mean_eta=prior_mean_eta,prior_prec_eta=prior_prec_eta)
        inits<-function() (list(alpha=rep(0,A),Beta_rest=rep(1/A,A-1),beta_rest_mat=matrix(1/A,nrow=p-1,ncol=(A-1)),Kappa_rest=rep(0,T-1),kappa_rest_mat=matrix(0,nrow=p-1,ncol=(T-1)),rho_kappa=0.5,eta_kappa=c(0,0),i_sigma2_kappa=0.1,rho_Kappa=0.5,eta_Kappa=c(0,0),i_sigma2_Kappa=0.1))
        if (p==1){
          vars<-c("q","alpha","Beta","Kappa","eta_Kappa","rho_Kappa","sigma2_Kappa")
        } else {
          vars<-c("q","alpha","beta","kappa","Beta","Kappa","eta_kappa","rho_kappa","sigma2_kappa","eta_Kappa","rho_Kappa","sigma2_Kappa")
        }
        logit_LC_MLiLee_modified_alt_forecast_jags<-rjags::jags.model(system.file("models/logit_LC_MLiLee_modified_alt_forecast.jags",package="BayesMoFo"),data=data,inits=inits,n.chain=n.chain,n.adapt = n.adapt,quiet=quiet)
        result_jags<-rjags::coda.samples(logit_LC_MLiLee_modified_alt_forecast_jags,vars,n.iter=n_iter,thin=thin)
        
      }
      if (family=="poisson"){
        data<-list(dxt=death_forecast,ext=expo_forecast,A=A,T=T,p=p,h=h,prior_mean_Beta=prior_mean_Beta,prior_prec_Beta=prior_prec_Beta,prior_mean_beta=prior_mean_beta,prior_prec_beta=prior_prec_beta,matrix_kappa_X=matrix_kappa_X,prior_mean_eta=prior_mean_eta,prior_prec_eta=prior_prec_eta)
        inits<-function() (list(alpha=rep(0,A),Beta_rest=rep(1/A,A-1),beta_rest_mat=matrix(1/A,nrow=p-1,ncol=(A-1)),Kappa_rest=rep(0,T-1),kappa_rest_mat=matrix(0,nrow=p-1,ncol=(T-1)),rho_kappa=0.5,eta_kappa=c(0,0),i_sigma2_kappa=0.1,rho_Kappa=0.5,eta_Kappa=c(0,0),i_sigma2_Kappa=0.1))
        if (p==1){
          vars<-c("q","alpha","Beta","Kappa","eta_Kappa","rho_Kappa","sigma2_Kappa")
        } else {
          vars<-c("q","alpha","beta","kappa","Beta","Kappa","eta_kappa","rho_kappa","sigma2_kappa","eta_Kappa","rho_Kappa","sigma2_Kappa")
        }
        log_LC_MLiLee_modified_alt_forecast_jags<-rjags::jags.model(system.file("models/log_LC_MLiLee_modified_alt_forecast.jags",package="BayesMoFo"),data=data,inits=inits,n.chain=n.chain,n.adapt = n.adapt,quiet=quiet)
        result_jags<-rjags::coda.samples(log_LC_MLiLee_modified_alt_forecast_jags,vars,n.iter=n_iter,thin=thin)
      }
      if (family=="nb"){
        data<-list(dxt=death_forecast,ext=expo_forecast,A=A,T=T,p=p,h=h,prior_mean_Beta=prior_mean_Beta,prior_prec_Beta=prior_prec_Beta,prior_mean_beta=prior_mean_beta,prior_prec_beta=prior_prec_beta,matrix_kappa_X=matrix_kappa_X,prior_mean_eta=prior_mean_eta,prior_prec_eta=prior_prec_eta)
        inits<-function() (list(alpha=rep(0,A),Beta_rest=rep(1/A,A-1),beta_rest_mat=matrix(1/A,nrow=p-1,ncol=(A-1)),Kappa_rest=rep(0,T-1),kappa_rest_mat=matrix(0,nrow=p-1,ncol=(T-1)),rho_kappa=0.5,eta_kappa=c(0,0),i_sigma2_kappa=0.1,rho_Kappa=0.5,eta_Kappa=c(0,0),i_sigma2_Kappa=0.1,phi=100))
        if (p==1){
          vars<-c("q","alpha","Beta","Kappa","eta_Kappa","rho_Kappa","sigma2_Kappa","phi")
        } else {
          vars<-c("q","alpha","beta","kappa","Beta","Kappa","eta_kappa","rho_kappa","sigma2_kappa","eta_Kappa","rho_Kappa","sigma2_Kappa","phi")
        }
        nb_LC_MLiLee_modified_alt_forecast_jags<-rjags::jags.model(system.file("models/nb_LC_MLiLee_modified_alt_forecast.jags",package="BayesMoFo"),data=data,inits=inits,n.chain=n.chain,n.adapt = n.adapt,quiet=quiet)
        result_jags<-rjags::coda.samples(nb_LC_MLiLee_modified_alt_forecast_jags,vars,n.iter=n_iter,thin=thin)
      }
    }
  }
  
  invisible(gc())
  list(post_sample=result_jags,param=vars[-1],death=death,expo=expo,family=family,forecast=forecast,h=h)
  
}

fit_MLiLee_sharealpha<-function(death,expo,n_iter=10000,family="nb",
                                n.chain=1,thin=1,n.adapt=1000,forecast=FALSE,h=5,quiet=FALSE){
  
  fit_MLiLee(death=death,expo=expo,share_alpha=TRUE,n_iter=n_iter,family=family,
             n.chain=n.chain,thin=thin,n.adapt=n.adapt,quiet=quiet,forecast=forecast,h=h)
  
}

